// VTest.cpp -- A test shell for the Vaunix LabView dll
//
//	RSD 11/2008 -- new version that uses the dll
//
//	V.9		-- Uses external dll for all device IO
//

#include "stdafx.h"

#include "vnx_fsynsth.h"

// ------------------------------- Equates -----------------------------------------------
#define SWP_DIRECTION		0x04	// MASK: bit = 0 for sweep up, 1 for sweep down 
#define SWP_CONTINUOUS		0x02	// MASK: bit = 1 for continuous sweeping
#define SWP_ONCE			0x01	// MASK: bit = 1 for single sweep
 

// ------------------------------- Allocations -------------------------------------------

static DEVID MyDevices[MAXDEVICES];			// I have statically allocated this array for convenience
											// It holds a list of device IDs for the connected devices
											// They are stored starting at MyDevices[0]

static char MyDeviceName[MAX_MODELNAME];	// NB -- its a char array for now, no support for wchars, etc.

static string sDevName			= "LSG-402";// device name string
static bool gbWantOneDevice		= FALSE;

static long DevNum				= 0;		// which device we should work with

static string sDevFrequency		= "";
static long Frequency			= 601	;	// default test frequency 60 MHz

static string sDevFStart		= "";
static long FStart				= 600;		// default test start frequency

static string sDevFStop			= "";
static long FStop				= 1003;		// default test stop is 100 Mhz

static string sDevDwell			= "";
static long Dwell				= 1004;		// default dwell time is 1 second

static string sDevFStep			= "";
static long FStep				= 10;		// default step size is 10Mhz

static string sDevAtten			= "";
static long Atten				= 0;		// default attenuation is 0db, encoding is in .25 db steps

static string sDevPower			= "";
static long Power				= 4;		// sets output power level, relative to device's max calibrated power
											// it is in units of .25 db of attenuation - 00 is the largest signal

static long SerialNumber		= 0;		// used to hold the serial number for the get serial number command

static int RefOsc				= 0;		// really used as a bool -- if non zero, then we use the internal osc.

static int RFOnOff				= 1;		// really used as a bool -- if non zero, turn on the RF output

static int Sweep_mode			= 0;		// the low byte is the sweep command byte
static int GetParam				= 0;		// the low byte is the GET command byte

bool gbWantSetFrequency			= FALSE;
bool gbWantSetFStart			= FALSE;
bool gbWantSetFStop				= FALSE;
bool gbWantSetDwell				= FALSE;
bool gbWantStartSweep			= FALSE;
bool gbWantSetFStep				= FALSE;
bool gbWantSetAtten				= FALSE;
bool gbWantSetPower				= FALSE;
bool gbWantSaveSettings			= FALSE;
bool gbWantSetSerialNum			= FALSE;
bool gbWantGetParam				= FALSE;
bool gbGotReply					= FALSE;
bool gbWantMaxPower				= FALSE;
bool gbBatchMode				= FALSE;
bool gbWantSetRefOsc			= FALSE;
bool gbWantSetRFOnOff			= FALSE;

// ------------------------------- Support Routines --------------------------------------

void PrintHelp()
{
	printf("VTest\n");
	printf("\n");
//	printf("Hit CTRL+C to exit\n");
	printf("\n");

	printf(" --- Overall modes and device selection. Defaults to all devices ---\n");
	printf(" This version controls the first Lab Brick Synthesizer it finds\n");
//	printf("  -b        Batch mode, exit after displaying one status report\n");
	printf("  -d        Device Number -- 1 to NDevices\n");
	printf("\n");

	printf(" --- Commands to set parameters and start sweep --- \n");
//	printf("  -a nn     Set attenuator, nn is attenuation in .25db units\n");
	printf("  -f nn     Set frequency, nn is frequency in 100Khz units\n");
	printf("  -s nn     Set sweep start frequency, nn is start frequency\n");
	printf("  -e nn     Set sweep end frequency, nn is end frequency\n");
	printf("  -t nn     Set time to dwell on frequency, nn is time in ms.\n");
	printf("  -i nn     Set frequency increment, nn is the frequency increment\n");
	printf("            in 100khz units.\n");
	printf("  -p nn     Set output power, nn is db of extra attenuation,\n");
	printf("             in .25 db steps. 1 db = 4 \n");
	printf("  -g n      Start a sweep, 1 = once upwards, 2 = continuous upwards\n");
	printf("             5 = once down, 6 = continuous down\n");
	printf("  -o n      Select the reference oscillator to use -- 1 = internal, 0 = external\n");
	printf("  -r n      Turn the RF output on or off -- 1 = on, 0 = off\n");
	
	printf("  -y        Write user settings to flash\n");
	printf("\n");

#if 0
	printf(" --- Query command to read values --- \n");
	printf("  -q n      GET a value with selector = n:\n");
	printf("             Frequency = 4\n");
	printf("             Dwell Time = 5\n"); 
	printf("             Sweep Start = 6\n");
	printf("             Sweep End = 7\n");
	printf("             Frequency Step = 8\n");
	printf("             Sweep Mode = 9\n");
	printf("             RF Enable = 10\n");
	printf("             Attenuator Setting = 11\n");
	printf("             Power Setting = 13\n");
	printf("             Minimum Frequency = 32\n");
	printf("             Maximum Frequency = 33\n");
	printf("             MaxPower = 27\n");
	printf("             Serial Number = 31\n");
	printf("             Model Name = 34\n");
	printf("             Int Ref Enabled = 35\n");

#endif
}

// --------------------- MakeLower ------------------------------

char MakeLower(char &c)
{
	return c = tolower(c);
}

// --------------------------------------------------------------




// ---------- ParseCommandLine ----------------------------------------------- 

// ParseCommandLine() will return FALSE to indicate that we received an invalid
// command or should abort for another reason.

bool ParseCommandLine(int argc, char *argv[])
{
	enum { wantDash, wantDevNumber, wantFrequency, wantFStart, wantFStop, wantDwell, wantFStep, wantAtten, wantPower, wantIndex, wantHiLo, wantSerialNum, wantSweep, wantGetParam, wantMaxPower, wantSetRFOnOff, wantSetRefOsc } state = wantDash;

	for (int i = 1; i < argc; ++i) {
		// Convert each argument to lowercase
		string thisParam(argv[i]);
		for_each(thisParam.begin(), thisParam.end(), MakeLower);

		bool bHasDash = ('-' == thisParam[0]);
		if (bHasDash) {
			if (state != wantDash) {
				PrintHelp();
				return FALSE;
			}

			// remove the dash from the front of the string
			thisParam = string(thisParam.begin() + 1, thisParam.end());

			// Identify the argumenets
			if ("d" == thisParam) {
				// -d should be followed by a number
				state = wantDevNumber;
			}
			else if ("b" == thisParam) {
				gbBatchMode = TRUE;
			}
			else if ("f" == thisParam) {
				gbWantSetFrequency = TRUE;
				state = wantFrequency;
			}
			else if ("s" == thisParam) {
				gbWantSetFStart = TRUE;
				state = wantFStart;
			}
			else if ("e" == thisParam) {
				gbWantSetFStop = TRUE;
				state = wantFStop;
			}
			else if ("t" == thisParam) {
				gbWantSetDwell = TRUE;
				state = wantDwell;
			}
			else if ("i" == thisParam) {
				gbWantSetFStep = TRUE;
				state = wantFStep;
			}
			else if ("a" == thisParam) {
				gbWantSetAtten = TRUE;
				state = wantAtten;
			}
			else if ("p" == thisParam) {
				gbWantSetPower = TRUE;
				state = wantPower;
			}
			else if ("m" == thisParam) {
				gbWantMaxPower = TRUE;
				state = wantMaxPower;
			}
			else if ("g" == thisParam) {
				gbWantStartSweep = TRUE;
				state = wantSweep;
			}
			else if ("y" == thisParam) {
				gbWantSaveSettings = TRUE;
				state = wantDash;
			}
			else if ("q" == thisParam) {
				gbWantGetParam = TRUE;
				state = wantGetParam;
			}
			else if ("r" == thisParam) {
				gbWantSetRFOnOff = TRUE;
				state = wantSetRFOnOff;
			}
			else if ("o" == thisParam) {
				gbWantSetRefOsc = TRUE;
				state = wantSetRefOsc;
			}
			else {
				// this case is for "-h" and any argument we don't recognize
				PrintHelp();
				return FALSE;	// don't continue
			}
		} else {
			// assert(state != wantDash);

			// save the whole substring and do conversions for each argument type

			switch (state){

			case wantDevNumber:
				DevNum = atol(thisParam.begin());
				state = wantDash;	// we always go back to the wantDash state to look for the next arg.
				break;
			
			case wantFrequency:
				Frequency = atol(thisParam.begin());		// convert to a long
				state = wantDash;
				break;

			case wantFStart:
				FStart = atol(thisParam.begin());
				state = wantDash;
				break;

			case wantFStop:
				FStop = atol(thisParam.begin());
				state = wantDash;
				break;

			case wantDwell:
				Dwell = atol(thisParam.begin());
				state = wantDash;
				break;

			case wantFStep:
				FStep = atol(thisParam.begin());
				state = wantDash;
				break;

			case wantAtten:
				Atten = atol(thisParam.begin());
				state = wantDash;
				break;

			case wantPower:
				Power = atol(thisParam.begin());
				state = wantDash;
				break;


			case wantSerialNum:
				SerialNumber = atol(thisParam.begin());
				state = wantDash;
				break;

			case wantSweep:
				Sweep_mode = atoi(thisParam.begin());
				state = wantDash;
				break;

			case wantSetRFOnOff:
				RFOnOff = atoi(thisParam.begin());
				state = wantDash;
				break;			
			
			case wantSetRefOsc:
				RefOsc = atoi(thisParam.begin());
				state = wantDash;
				break;

			case wantGetParam:
				GetParam = atoi(thisParam.begin());
				state = wantDash;
				break;
			}


		}
	}

	if (state != wantDash) {
		// we are expecting an argument, if we didn't get one then print the help message
		PrintHelp();
		return FALSE;
	}

	// It's OK to continue
	return TRUE;
}











// -------------------------------- Program Main -----------------------------------------

int main(int argc, char* argv[])
{
	int i, j;
	int itemp;
	bool bTemp;
	float ftemp;
	float fMaxPwr;
	float fMinPwr;

	printf("Lab Brick Frequency Synthesizer Test Program\n");

	if (!ParseCommandLine(argc, argv))
		return 0;

	DevNum = DevNum - 1;
	if (DevNum < 0) DevNum = 0;


	// --- if TestMode = TRUE then the dll will fake the hardware ---
	fnLSG_SetTestMode(FALSE);
	
	i = fnLSG_GetNumDevices();

	if (i == 0){
		printf("No device found\n");
	}

	if (i == 1){
		printf("Found %d Device\n", i);

	}
	else {
		printf("Found %d Devices\n", i);
	}

	i = fnLSG_GetDevInfo(MyDevices);

	printf("Got Device Info for %d Device[s]\n", i);

	
	
	
	if ( i > 0 )	// do we have a device?
	{
		for (j = 0; j < i; j++){

		// --- print out the first device's name ---
		itemp = fnLSG_GetModelName(MyDevices[j], MyDeviceName);
		printf("Device %d is an %s \n", i, MyDeviceName);

		// --- print out the device's serial number ---
		itemp = fnLSG_GetSerialNumber(MyDevices[j]);
		printf("Device %d has serial number %d \n", i, itemp);


		// --- We need to init the device (open it) before we can do anything else ---
		itemp = fnLSG_InitDevice(MyDevices[j]);

		if (itemp){
			printf("InitDevice returned %x\n", itemp);
		}
		
		// --- Lets see if we got the device's parameters ---
		
		itemp = fnLSG_GetStartFrequency(MyDevices[j]);
		printf("Sweep Start Frequency = %d\n", itemp);
		
		itemp = fnLSG_GetEndFrequency(MyDevices[j]);
		printf("Sweep End Frequency = %d\n", itemp);
		
		itemp = fnLSG_GetDwellTime(MyDevices[j]);
		printf("Sweep Dwell Time = %d\n", itemp);
		
		itemp = fnLSG_GetFrequencyStep(MyDevices[j]);
		printf("Sweep FrequencyStep Size = %d\n", itemp);

		itemp = fnLSG_GetMinFreq(MyDevices[j]);
		printf("Minimum Frequency = %d\n", itemp);
		
		itemp = fnLSG_GetMaxFreq(MyDevices[j]);
		printf("Maximum Frequency = %d\n", itemp);

		itemp = fnLSG_GetMinPwr(MyDevices[j]);
		fMinPwr = itemp * .25;	// we represent power levels in .25db steps
		printf("Minimum Output Power Level = %f db\n", fMinPwr);

		itemp = fnLSG_GetMaxPwr(MyDevices[j]);
		fMaxPwr = itemp * .25;	// we represent power levels in .25db steps
		printf("Maximum Output Power Level = %f db\n", fMaxPwr);


		// --- Show if the RF output is on ---
		itemp = fnLSG_GetRF_On(MyDevices[j]);

		if (itemp != 0){
			printf("RF ON\n");
		}
		else{
			printf("RF OFF\n");
		}

		// --- Show Ref. Oscillator Source ---
		itemp = fnLSG_GetUseInternalRef(MyDevices[j]);

		if (itemp != 0){
			printf("Using Internal Reference Frequency\n");
		}
		else{
			printf("Using External Reference Frequency\n");
		}

		itemp = fnLSG_GetPowerLevel(MyDevices[j]);
		ftemp = itemp * .25;	// we represent power levels in .25db steps
		printf("Output Power Attenuation = %f db\n", ftemp);



		} // end of the for loop over the devices

		// ------------- Now we'll set the requested device with new parameters -------------

		// --- Lets try to set the frequency for starters.. ---

		if (gbWantSetFrequency)
		{
			itemp = fnLSG_SetFrequency(MyDevices[DevNum], Frequency);
		}

		// --- and then do whatever else the user requested ---
		
		if (gbWantSetDwell)
		{
			itemp = fnLSG_SetDwellTime(MyDevices[DevNum], Dwell);
		}

		if (gbWantSetFStart)
		{
			itemp = fnLSG_SetStartFrequency(MyDevices[DevNum], FStart);
		}
	
		if (gbWantSetFStop)
		{
			itemp = fnLSG_SetEndFrequency(MyDevices[DevNum], FStop);
		}

		if (gbWantSetFStep)
		{
			itemp = fnLSG_SetFrequencyStep(MyDevices[DevNum], FStep);
		}

		if (gbWantSetPower)
		{
			itemp = fnLSG_SetPowerLevel(MyDevices[DevNum], Power);
		}

		if (gbWantStartSweep)
		{
			// --- first we'll figure out what the user wants us to do ---
			if (Sweep_mode & SWP_DIRECTION)
			{ bTemp = FALSE;}
			else
			{ bTemp = TRUE;}	// NB -- the flag is TRUE for "up" in the SetSweepDirection call.
								// but the old test program uses a 0 bit for up, and a 1 bit for down...

			itemp = fnLSG_SetSweepDirection(MyDevices[DevNum], bTemp);

			// --- and now we'll do the mode - one time sweep or repeated sweep ---

			if (Sweep_mode & SWP_ONCE)
			{ bTemp = FALSE;}
			else
			{ bTemp = TRUE;}	// NB -- the flag is TRUE for "repeated" in the SetSweepMode call.
								// but the old test program encodes the modes differently

			itemp = fnLSG_SetSweepMode(MyDevices[DevNum], bTemp);


			printf("Starting a Frequency Sweep\n");
			itemp = fnLSG_StartSweep(MyDevices[DevNum], TRUE);
		}


		if (gbWantSaveSettings)
		{
			fnLSG_SaveSettings(MyDevices[DevNum]);
		}

		if (gbWantSetRFOnOff)
		{

			if (RFOnOff == 0)
			{	bTemp = FALSE;}
			else
			{	bTemp = TRUE;}

			itemp = fnLSG_SetRFOn(MyDevices[DevNum], bTemp);
		}


		if (gbWantSetRefOsc)
		{

			if (RefOsc == 0)
			{	bTemp = FALSE;}
			else
			{	bTemp = TRUE;}

			itemp = fnLSG_SetUseInternalRef(MyDevices[DevNum], bTemp);
		}

		
		// -- Lets hang around some and report on the device's operation
		
		j = 0;

		while (j < 20)
		{

			itemp = fnLSG_GetFrequency(MyDevices[DevNum]);
			printf("Frequency = %d\n", itemp);

			itemp = fnLSG_GetPowerLevel(MyDevices[DevNum]);
			ftemp = itemp * .25;	// we represent power levels in .25db steps
			printf("Output Power Level = %f db\n", (fMaxPwr - ftemp));

			Sleep(500);		// wait for 1/2 second
			
			j++;
		
		}
		
		// -- we've done whatever the user wanted, time to close the devices

		printf("Closing devices...\n");
		
		for (j = 0; j < i; j++)
		{
			itemp = fnLSG_CloseDevice(MyDevices[j]);

		}




	} // end of if ( i > 0 ) -- "we have a device"






	return 0;
}

