//
//  main.c
//  LSW_Example
//
// ---------- Demonstration Program ------------------------------------------------------
//
// This program shows how to call the RF switch functions
//
// It is not intended as an example of appropriate user interface design, or the error handling
// that would be present in a production program.
// Several examples of testing for error codes returned by DLL function calls are included, but for simplicity
// many example calls do not test for error cases that might be returned.
//
// Note that the example code gets the state of the switch hardware before the commands are sent to it, running
// the program again will display the settings of the device which resulted from the last run of the program.


#include <stdio.h>
#include <stdbool.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

#include "vnx_lsw_api.h"
#include <unistd.h>

// ------------------------------- Equates -----------------------------------------------
#define SWPAT_REPEAT        0x02    // MASK: bit = 1 for continuous sweeping
#define SWPAT_ONETIME       0x01    // MASK: bit = 1 for single sweep

// ------------------------------- Allocations -------------------------------------------
static DEVID MyDevices[MAXDEVICES];     // I have statically allocated this array for convenience
                                        // It holds a list of device IDs for the connected devices
                                        // They are stored starting at MyDevices[0]

static char MyDeviceNameA[MAX_MODELNAME];       // NB -- this is a single byte char array for testing the ASCII name function
static wchar_t MyDeviceNameW[MAX_MODELNAME];    // NB -- this is a WCHAR array for testing the Unicode name function

static char errmsg[32];                         // For the status->string converter
static char cModelName[32];                     // buffer for the model name

static bool gbWantOneDevice = FALSE;

static int DevNum = 0;              // the device we should work with.
static int DevRange = 1;            // the number of devices we'll send the command to
static int NumDevices = 0;          // used to store the actual number of devices found

// --------------------------- Variables -------------------------------------------------
static int SerialNumber = 0;            // used to hold the serial number for the get serial number command
static int Switch_Select = 0;           // used to hold the switch number to select
static int Set_External_Control = 0;    // used to choose internal or external switch control signals
static int Use_Pulse_Mode = 0;          // used to enable pulsed mode operation if 1
static int Start_Pattern = 0;           // used to start the pattern if 1

static float ActivePulseTime = 1.0;     // default active pulse time is 1 second
static float InactivePulseTime = 1.0;   // default inactive pulse time is also 1 second

static int PatternIndex[MAX_PATTERN_LENGTH];     // the step number in the pattern (0 to 7)
static int PatternSwitch[MAX_PATTERN_LENGTH];    // the switch setting for the pattern step
static int PatternDwellTime[MAX_PATTERN_LENGTH]; // how long to stay on the pattern step

bool gbQuietMode = FALSE;               // true if the user wants less printouts
bool gbWantSwitch = FALSE;              // the user wants to set the switch
bool gbWantExtControl = FALSE;          // the user wants to set internal vs external control signals
bool gbWantSaveSettings = FALSE;        // the user wants to save the settings
bool gbWantPulseMode = FALSE;           // the user wants to start or stop pulsed mode
bool gbWantGetParam = FALSE;            // the user wants to get a parameter
bool gbWantSwitchPattern = FALSE;
bool gbGotReply = FALSE;
bool gbBatchMode = FALSE;               // the user wants to exit immediately after setting parameters in the Lab Brick
bool gbWantPulseActive = FALSE;         // the user wants to set the length of the active pulse interval
bool gbWantPulseInactive = FALSE;       // the user wants to set the length of the inactive pulse interval
bool gbWantSetPatternElement = FALSE;   // the user wants to set one or more pattern elements



// local function declarations
void PrintHelp(void);
void MakeLower(char *cstr);
char* fnLSW_perror(LVSTATUS status);
void CheckAPISet(LVSTATUS status);
void CheckAPI_Float(float test);
char* fnLSW_pstatus(LVSTATUS status);

// ------------------------------- Support Routines --------------------------------------
void PrintHelp(void)
{
    printf("Vaunix RF Switch Demo\n");
    printf("\n");
    printf("Hit CTRL+C to exit\n");
    printf("\n");
    
    printf(" --- Overall modes and device selection. Defaults to first device ---\n");
    printf(" Each command can only appear once on the command line, except for the -v command to set pattern elements\n");
    printf("  -d i n     Select the devices to work with, i is the device number (1,2,3, etc.)\n");
    printf("             and n is the number of devices to apply the command to.\n");
    printf("             -d 1 2 applies the commands to RF switches 1 and 2.\n");
    printf("             -d 2 3 applies the commands to RF switches 2, 3 and 4.\n");
    printf("  -y        Save the current settings in the device or devices.\n");
    printf("\n");
    printf("  -b        Batch mode, exit immediately after sending commands to the Lab Bricks.\n");
    printf("  -q        Quiet mode, skip most outputs.\n");
    printf("\n");
    printf(" --- Commands to set parameters and start patterns or pulse mode --- \n");
    printf("  -s n      Set the switch, n is the active switch number\n");
    printf("  -x n      Select internal (n=0) or external (n=1) control signals\n");
    printf("  -a nn     Set the active pulse time - RFC => RF2.\n");
    printf("  -i nn     Set the inactive pulse time - RFC => RF1.\n");
    printf("  -p n      Start pulse mode, 1 = on, 0 to stop\n");
    printf("  -v i n t    Set an element in the pattern, i is the index, 0 to 7, n is the switch setting, 1 to 4, and t is the time in seconds\n");
    printf("  -g n      Start a switch pattern, 1 = once, 2 = repeating, 0 to stop\n");
    
    printf("\n");
}

// --------------------- MakeLower ------------------------------
void MakeLower(char *cstr)
{
    int i;
    for (i = 0; cstr[i]; i++){
        cstr[i] = tolower(cstr[i]);
    }
    return;
}

// --------------------------------------------------------------
#define MAX_MSG 32

// A function to display an error status as an ASCII string
char* fnLSW_perror(LVSTATUS status) {
    strcpy(errmsg, "STATUS_OK");
    if (status == BAD_PARAMETER) strcpy(errmsg, "BAD_PARAMETER");
    if (status == BAD_HID_IO) strcpy(errmsg, "BAD_HID_IO");
    if (status == DEVICE_NOT_READY) strcpy(errmsg, "DEVICE_NOT_READY");
    if (status == INVALID_DEVID) strcpy(errmsg, "INVALID_DEVID");
    
    return errmsg;
}

// -- one way to check for errors --
void CheckAPISet(LVSTATUS status)
{
    if (status & 0x80000000)
    {
        printf("*** Error: LSW API returned status = %x, %s ***\n", status, fnLSW_perror(status));
    }
}

// -- one way to check for errors for functions that return a float --
void CheckAPI_Float(float test)
{
    if (test == F_INVALID_DEVID )
    {
        printf("*** Error: LSW API returned status = %x, %s ***\n", INVALID_DEVID, fnLSW_perror(INVALID_DEVID));
    }
    else if (test == F_DEVICE_NOT_READY)
    {
        printf("*** Error: LSW API returned status = %x, %s ***\n", DEVICE_NOT_READY, fnLSW_perror(DEVICE_NOT_READY));
    }
    // all other values are not errors
    return;
        
}

// A function to convert the status to an ASCII string
char* fnLSW_pstatus(LVSTATUS status)
{
    strcpy(errmsg, "STATUS_OK");
    
    // Status returns for DevStatus
    if (status == INVALID_DEVID) strcpy(errmsg, "INVALID_DEVID");
    if (status == DEV_CONNECTED) strcpy(errmsg, "DEV_CONNECTED");
    if (status == DEV_OPENED) strcpy(errmsg, "DEV_OPENED");
    if (status == SWPAT_ACTIVE) strcpy(errmsg, "PATTERN_ACTIVE");
    if (status == SWPAT_REPEAT) strcpy(errmsg, "PATTERN_REPEATING");
    
    return errmsg;
}

// ParseCommandLine() will return FALSE to indicate that we received an invalid
// command or should exit for another reason.
bool ParseCommandLine(int argc, const char *argv[])
{
    int PatIndex;
    int PatSwitch;            // temporary holders for our pattern element arguments
    float PatTime;            // we'll convert it to an integer number of milliseconds
    char thisParam[256];      // a working copy of the current argument from the command line
    
    enum { wantDash, wantDevSubstring, wantDevID, wantDevRange, wantSwitch, wantExtControl,
           wantPulseMode, wantSwitchPattern, wantActive, wantInactive, wantPatternIndex,
           wantPatternSetting, wantPatternDwellTime }
    
    state = wantDash;
    
    for (int i = 1; i < argc; ++i) {
        
        // Convert each argument to lowercase
        strcpy(thisParam, argv[i]);
        MakeLower(thisParam);
        
        // determine what to do with this portion of the command line
        if (state == wantDash)
        {
            if ('-' != thisParam[0])
            {
                printf("\n *** Error in command line syntax *** \n");
                PrintHelp();
                return FALSE;
            }
            
            // Identify the command line arguments
            if (strcmp(thisParam, "-d") == 0){
                state = wantDevID;
            }
            else if (strcmp(thisParam, "-x") == 0){
                gbWantExtControl = TRUE;
                state = wantExtControl;
            }
            else if (strcmp(thisParam, "-s") == 0){
                gbWantSwitch = TRUE;
                state = wantSwitch;
            }
            else if (strcmp(thisParam, "-a") == 0){
                gbWantPulseActive = TRUE;
                state = wantActive;
            }
            else if (strcmp(thisParam, "-i") == 0){
                gbWantPulseInactive = TRUE;
                state = wantInactive;
            }
            else if (strcmp(thisParam, "-p") == 0){
                gbWantPulseMode = TRUE;
                state = wantPulseMode;
            }
            else if (strcmp(thisParam, "-g") == 0){
                gbWantSwitchPattern = TRUE;
                state = wantSwitchPattern;
            }
            else if (strcmp(thisParam, "-y") == 0){
                gbWantSaveSettings = TRUE;
                state = wantDash;
            }
            else if (strcmp(thisParam, "-b") == 0){
                gbBatchMode = TRUE;
                state = wantDash;
            }
            else if (strcmp(thisParam, "-q") == 0){
                gbQuietMode = TRUE;
                state = wantDash;
            }
            else if (strcmp(thisParam, "-v") == 0){
                gbWantSetPatternElement = TRUE;
                state = wantPatternIndex;
            }
            else {
                // this case is for "-?" and any argument we don't recognize
                PrintHelp();
                return FALSE;    // don't continue
            }
        }
        
        else {
            
            // save the whole substring and do conversions for each argument type
            switch (state){
                    
                case wantSwitch:
                    Switch_Select = atoi(thisParam);    // get the switch number (1,2,3,4)
                    state = wantDash;
                    break;
                    
                case wantExtControl:
                    Set_External_Control = atoi(thisParam);   // convert to an int
                    state = wantDash;
                    break;
                    
                case wantActive:
                    ActivePulseTime = atoi(thisParam);
                    state = wantDash;
                    break;
                    
                case wantDevID:
                    DevNum = atoi(thisParam);
                    state = wantDevRange;
                    break;
                    
                case wantDevRange:
                    DevRange = atoi(thisParam);
                    state = wantDash;
                    break;
                    
                case wantInactive:
                    InactivePulseTime = atoi(thisParam);
                    state = wantDash;
                    break;
                    
                case wantPulseMode:
                    Use_Pulse_Mode = atoi(thisParam);
                    state = wantDash;
                    break;
                    
                case wantSwitchPattern:
                    Start_Pattern = atoi(thisParam);
                    state = wantDash;
                    break;
                    
                case wantPatternIndex:
                    PatIndex = atoi(thisParam);
                    if (PatIndex < 0 || PatIndex > MAX_PATTERN_LENGTH - 1)
                    {
                        printf("Invalid index value for pattern");
                        return FALSE;
                    }
                    state = wantPatternSetting;
                    break;
                    
                case wantPatternSetting:
                    PatSwitch = atoi(thisParam);
                    if (PatSwitch < 1 || PatSwitch > 4)                // note - we don't know yet if this is valid for the HW we have
                    {
                        printf("Invalid switch value for pattern");
                        return FALSE;
                    }
                    state = wantPatternDwellTime;
                    break;
                    
                case wantPatternDwellTime:
                    PatTime = (float) atof(thisParam);
                    if (PatTime < .001 || PatTime > 1.0e+4)                // note - HW limit is larger, just a likely range check here
                    {
                        printf("Out of range dwell time value for pattern");
                        return FALSE;
                    }
                    // now we can add the user's pattern element to our collection of them
                    PatternIndex[PatIndex] = PatIndex;
                    PatternSwitch[PatIndex] = PatSwitch;
                    PatternDwellTime[PatIndex] = (int)PatTime * 1000;    // our units are milliseconds
                    state = wantDash;
                    break;
                    
            }
        }
    }
    
    if (state != wantDash) {
        // we are expecting an argument, if we didn't get one then print the help message
        PrintHelp();
        return FALSE;
    }
    
    // It's OK to continue
    return TRUE;
}
// ------------- Command Line Main ---------------------------------------------------

int main(int argc, const char * argv[])
{
    int i, j, k;
    int iDev;
    int itemp, itemp2, n_entries;
    bool bTemp;
    float ftemp, PulseOnTime, PulseRepTime;
    
    itemp = 0;                  // clear a warning...
    
    if (!ParseCommandLine(argc, argv))
        return 0;
    
    if (!gbQuietMode) printf("Lab Brick RF Switch API Demonstration Program\n");
    
    // -- convert the user's device number to our internal MyDevices array index and check our device range --
    DevNum = DevNum - 1;
    
    if (DevNum < 0) DevNum = 0;
    if (DevRange < 1) DevRange = 1;
    if (DevRange > MAXDEVICES) DevRange = MAXDEVICES;
    
    if (DevNum  > MAXDEVICES - 1) DevNum = MAXDEVICES - 1;
    if ((DevNum + DevRange) > MAXDEVICES) DevRange = MAXDEVICES - DevNum;
    
    // at this point our device starting index and number of devices should be reasonable...
    if (!gbQuietMode) printf("Starting device number = %d, using %d device[s]\n", DevNum + 1, DevRange);
    
    // --- if TestMode = TRUE then the dll will fake the hardware ---
    fnLSW_SetTestMode(FALSE);
    
    // --- Use the tracing control function to turn on debug messages
    fnLSW_SetTraceLevel(0, 0, false);
    //fnLSW_SetTraceLevel(1, 1, true);
    //fnLSW_SetTraceLevel(-4, -4, true);
    
    i = fnLSW_GetNumDevices();
    
    if (i == 0){
        printf("No device found\n");
    }
    
    if (i == 1){
        if (!gbQuietMode) printf("Found %d Device\n", i);
    }
    else {
        if (!gbQuietMode) printf("Found %d Devices\n", i);
    }
    
    // -- warn the user if he or she expects more devices than we have --
    if (DevRange > i){
        printf(" Warning - not enough LSW devices are connected\n");
    }
    
    NumDevices = fnLSW_GetDevInfo(MyDevices);
    
    if (!gbQuietMode) printf("Got Device Info for %d Device[s]\n", NumDevices);
    
    if (NumDevices > 0)    // do we have a device?
    {
        for (j = 0; j < NumDevices; j++){
            
            // --- print out the first device's name ---
            if (!gbQuietMode){
                itemp = fnLSW_GetModelNameA(MyDevices[j], MyDeviceNameA);
                printf("Device %d is an %s \n", j + 1, MyDeviceNameA);
            }
            
            // --- print out the device's serial number ---
            if (!gbQuietMode){
                itemp = fnLSW_GetSerialNumber(MyDevices[j]);
                if (itemp >= 0)
                    printf("Device %d has serial number %d \n", j + 1, itemp);
            }
            
            
            // --- We need to init the device (open it) before we can do anything else ---
            itemp = fnLSW_InitDevice(MyDevices[j]);
            
            if (itemp){
                printf("InitDevice returned error code %x\n", itemp);
            }
            
            // --- Lets see if we got the device's parameters ---
            if (!gbQuietMode) {
                itemp = fnLSW_GetNumSwitches(MyDevices[j]);

                if (itemp >= 0)
                    printf("There are %d switches\n", itemp);
                else
                    CheckAPISet(itemp);
                
                itemp = fnLSW_GetSwitchSetting(MyDevices[j]);
                if (itemp >= 0)
                    printf("Switch %d is selected\n", itemp);
                else
                    CheckAPISet(itemp);
                
                itemp = fnLSW_GetActiveSwitch(MyDevices[j]);
                if (itemp >= 0)
                    printf("Switch %d is active\n", itemp);
                else
                    CheckAPISet(itemp);
                
                // --- We'll see what the switch pattern type was last set to ---
                itemp = fnLSW_GetPatternType(MyDevices[j]);
                if (itemp == 0)
                    printf("No pattern active\n");
                else if (itemp & SWPAT_REPEAT)
                    printf("Repeating pattern mode\n");
                else if (itemp & SWPAT_ONETIME)
                    printf("Single pattern mode\n");
                else
                    CheckAPISet(itemp);
                
                // --- and some of the other pattern values ---
                n_entries = fnLSW_GetPatternLength(MyDevices[j]);
                CheckAPISet(itemp);
                if (n_entries >= 0)
                    printf("There are %d entries in the pattern\n", n_entries);
                
                for (k = 0; k < n_entries; k++)
                {
                    itemp = fnLSW_GetPatternEntryTime(MyDevices[j], k);
                    printf("Hold time for switch pattern entry %d is %d \n", k, itemp);

                    itemp = fnLSW_GetPatternEntrySwitch(MyDevices[j], k);
                    printf("Switch setting for switch pattern entry %d is %d \n", k, itemp);

                } // end of for loop over pattern entries
                
                // --- and print out some other settings from the device ---
                itemp = fnLSW_GetUseExternalControl(MyDevices[j]);
                if (itemp < 0)
                    CheckAPISet(itemp);
                else if (itemp > 0)
                    printf("Using external control signals\n");
                else
                    printf("Using internal control signals\n");
                
                
                ftemp = fnLSW_GetPulseOnTime(MyDevices[j]);
                CheckAPI_Float(ftemp);
                if (ftemp >= 0)
                    printf("Pulse On Time = %g seconds, ", ftemp);

                ftemp = fnLSW_GetPulseOffTime(MyDevices[j]);
                CheckAPI_Float(ftemp);
                if (ftemp >= 0)
                    printf("Pulse Off Time = %g seconds\n", ftemp);
                
                // --- get the operation mode for pulsed operation, showing quick testing for errors in the return value

                itemp = fnLSW_GetPulseMode(MyDevices[j]);
                if (itemp & STATUS_ERROR){
                    printf("fnLSW_GetPulseMode returned error code %x\n", itemp);
                }
                else {
                    if (itemp) printf("Pulsed mode on\n");
                    else printf("Pulsed mode off\n");
                }

                itemp = fnLSW_GetHasFastPulseMode(MyDevices[j]);
                if (itemp & STATUS_ERROR){
                    fnLSW_perror(itemp);
                    printf("fnLSW_GetHasFastPulseMode returned error code %x %s\n", itemp, errmsg);
                }
                else {
                    if (itemp) printf("Pulsed mode available\n");
                    else printf("Pulsed mode not available\n");
                }
                
                printf("\n");
                
            } // end of our quiet mode case
            
        } // end of the for loop over the devices
        
        // if the user is trying to control a device we don't have, then quit now
        if (DevNum > NumDevices - 1){
            for (j = 0; j < NumDevices; j++)
            {
                itemp = fnLSW_CloseDevice(MyDevices[j]);
            }
            printf("First selected device is not attached, exiting.\n");
            return 0;            // quit - nothing else to do
        }
        
        // if the user is trying to control more devices than we have, reduce the number of devices in the group
        if ((DevNum + DevRange) > NumDevices){
            DevRange = NumDevices - DevNum;
            printf("Not enough LSW devices connected, using %d devices.\n", DevRange);
        }
        
        // ------------- Now we'll set the requested device or devices with new parameters -------------
        if (!gbQuietMode)printf("Setting the RF switch parameters..\n");
        
        for (iDev = DevNum; iDev < DevNum + DevRange; iDev++)
        {
            // --- Lets set the switch first ---
            if (gbWantSwitch)
            {
                if (!gbQuietMode) printf("Setting the switch position for device %d to %d\n", iDev + 1, Switch_Select);     // user switch setting
                itemp = fnLSW_SetSwitch(MyDevices[DevNum], Switch_Select);
                CheckAPISet(itemp);
            }
            
            // --- and then do whatever else the user requested ---
            
            if (gbWantExtControl)
            {
                if (Set_External_Control == 0)
                {
                    bTemp = FALSE;
                }
                else
                {
                    bTemp = TRUE;
                }

                itemp = fnLSW_SetUseExternalControl(MyDevices[DevNum], bTemp);
                CheckAPISet(itemp);
            }
            
            if (gbWantPulseMode)
            {
                // we'll use the values entered by the user, without error checking
                PulseOnTime = ActivePulseTime;
                PulseRepTime = ActivePulseTime + InactivePulseTime;

                // start the pulse modulation if the user wants to, otherwise stop pulse modulation
                if (Use_Pulse_Mode)
                {
                    itemp = fnLSW_SetFastPulsedOutput(MyDevices[DevNum], PulseOnTime, PulseRepTime, TRUE);
                    if (itemp)
                    {
                        fnLSW_perror(itemp);
                        printf("fnLSW_SetFastPulsedOutput returned %x, %s\n", itemp, errmsg);
                    }
                }
                else
                {
                    itemp = fnLSW_EnableInternalPulseMod(MyDevices[DevNum], FALSE);
                    if (itemp)
                    {
                        fnLSW_perror(itemp);
                        printf("fnLSW_EnableInternalPulseMod returned %x, %s\n", itemp, errmsg);
                    }
                }
            }

            if (gbWantSwitchPattern)
            {
                // start a pattern if the user wants to, otherwise stop whatever pattern is active
                if (Start_Pattern == 0)
                {
                    printf("Stopping the Switch Pattern\n");
                    itemp = fnLSW_StartPattern(MyDevices[DevNum], FALSE);
                    if (itemp) printf("fnLSW_StartPattern returned %x\n", itemp);
                }
                else
                {
                    // --- first we'll download the pattern entered by the user ---
                    bTemp = FALSE;
                    
                    for (j = 0; j < MAX_PATTERN_LENGTH; j++)
                    {
                        if (PatternIndex[j] > 0)                // make sure the user entered a pattern element for this index
                        {
                            // a zero dwell time entered by the user is used to indicate the end of a pattern
                            // in any event, we need an end of pattern marker at the end of the pattern, so we make sure that happens

                            if (PatternDwellTime[j] == 0 || j == MAX_PATTERN_LENGTH - 1) bTemp = TRUE;
                            
                            itemp = fnLSW_SetPatternEntry(MyDevices[DevNum], PatternSwitch[j], PatternDwellTime[j], j, bTemp);
                            
                            if (itemp)
                            {
                                fnLSW_perror(itemp);
                                printf("fnLSW_SetPatternEntry returned %x, %s\n", itemp, errmsg);
                            }
                        }

                    }

                    // --- and then we'll tell the hardware to execute the pattern ---
                    if (Start_Pattern == 2)
                    {
                        itemp = fnLSW_SetPatternType(MyDevices[DevNum], TRUE);    // the user wants a repeating pattern
                    }
                    else
                    {
                        itemp = fnLSW_SetPatternType(MyDevices[DevNum], FALSE);    // the user wants the pattern to run once

                    }

                    printf("Starting the Switch Pattern\n");
                    itemp = fnLSW_StartPattern(MyDevices[DevNum], TRUE);
                    if (itemp) printf("fnLSW_StartPattern returned %x\n", itemp);
                }
            }

            if (gbWantSaveSettings)
            {
                if (!gbQuietMode) printf("Saving the settings for device %d\n", iDev + 1);
                fnLSW_SaveSettings(MyDevices[iDev]);
                CheckAPISet(itemp);
            }
            
        }    // this is the end of our for loop over devices for the general commands
        

        // -- Lets report on the device's operation for a little while, unless we are in batch mode
        if (!gbBatchMode)
        {
            j = 0;
            while (j < 10)
            {
                for (iDev = DevNum; iDev < DevNum + DevRange; iDev++)
                {
                    itemp = fnLSW_GetActiveSwitch(MyDevices[DevNum]);
                    CheckAPISet(itemp);
                    if(itemp >= 0)
                        printf("Switch %d is active for device %d\n", itemp, iDev + 1);
                }
                printf("\n");
                usleep(500*1000);    // wait for 1/2 second
                j++;
            }
            
        } // end of if not batch mode
        
        // -- we've done whatever the user wanted, time to close the devices
        for (j = 0; j < i; j++)
        {
            itemp = fnLSW_CloseDevice(MyDevices[j]);
        }
        
    } // end of if ( i > 0 ) -- "we have a device"
    
    return 0;
}

// ===================== end of main ======================================



