#include <stdio.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdbool.h>
#include "lswdrvr.h"

#define DEBUG_LEVEL 1

// IP Address Validator
#define DELIM "."

// --------------------------- Global Variables -------------------------------------------------
static char gldadevicesip[16];   // device IP address (string form)
static int Swindex = 1;          // default to Switch-1
static int Swout = SW_PORT1;     // default output port

// Global flags for command line options
bool gbDeviceOpen        = false;
bool gbWantSaveSettings  = false;
bool gbWantSwitchStatus  = false;
bool gbWantSwitchCtrl    = false;
bool gbWantGetParam      = false;
bool gbQuietMode         = false;
bool gbBatchMode         = false;

//***************************************************************************
// Validate string of digits only
// Returns 1 if valid, 0 otherwise
//*****************************************************************************
static int valid_digit(const char *ip_str) {
    while (*ip_str) {
        if (*ip_str >= '0' && *ip_str <= '9')
            ++ip_str;
        else
            return 0;
    }
    return 1;
}

//***************************************************************************
// Validate IPv4 address string
// Returns 1 if valid dotted decimal IP (e.g. 192.168.1.1), else 0
//*****************************************************************************
static int is_valid_ip(const char *ip_str) {
    int num, dots = 0;
    char *ptr;
    char lstr[16];
    char *saveptr;

    // Copy into buffer so strtok_r doesn’t modify the input
    strncpy(lstr, ip_str, sizeof(lstr) - 1);
    lstr[sizeof(lstr) - 1] = '\0';

    // Break into tokens separated by "."
    ptr = strtok_r(lstr, DELIM, &saveptr);
    if (ptr == NULL) return 0;

    while (ptr) {
        if (!valid_digit(ptr))
            return 0;

        num = atoi(ptr);
        if (num >= 0 && num <= 255) {
            // Continue parsing remaining octets
            ptr = strtok_r(NULL, DELIM, &saveptr);
            if (ptr != NULL)
                ++dots;
        } else {
            return 0;
        }
    }

    // A valid IP must contain exactly 3 dots (4 parts)
    return (dots == 3);
}

//***************************************************************************
// Print program usage instructions
//*****************************************************************************
void PrintHelp(void) {
    printf("Vaunix RF Switch Demonstration\n\n");
    printf("Hit CTRL+C to exit\n\n");

    printf(" --- Overall modes and device selection based on ip address ---\n");
    printf("  -d i    Select the device by IP (192.168.100.11)\n");
    printf("  -r      Read the current information of the device\n");
    printf("  -y      Save the current settings in the device\n\n");

    printf("  -b      Batch mode, exit immediately after sending commands\n");
    printf("  -q      Quiet mode, skip most outputs\n\n");

    printf(" --- RF switch parameters --- \n");
    printf("  -g nn       Read RF Switch Port status of switch index nn\n");
    printf("  -s nn pp    Set RF Switch Port of switch index nn to port pp\n\n");
}

//***************************************************************************
// Parse command line arguments
// Sets global flags based on inputs
//*****************************************************************************
bool ParseCommandLine(int argc, char *argv[]) {
    enum {
        wantDash, wantDevID, wantSwindex, wantSwindexSet, wantSWRFout
    } state = wantDash;

    for (int i = 1; i < argc; ++i) {
        char *thisParam = argv[i];

        // Convert parameter string to lowercase in-place
        for (char *p = thisParam; *p; ++p)
            *p = (char)tolower(*p);

        if (state == wantDash) {
            // Must begin with dash
            if (thisParam[0] != '-') {
                printf("\n *** Error in command line syntax *** \n");
                PrintHelp();
                return false;
            }
            thisParam++; // skip '-'

            // Identify which argument we’re parsing
            if (strcmp(thisParam, "d") == 0) {
                state = wantDevID;
            } else if (strcmp(thisParam, "r") == 0) {
                gbWantGetParam = true;
            } else if (strcmp(thisParam, "g") == 0) {
                state = wantSwindex;
            } else if (strcmp(thisParam, "s") == 0) {
                state = wantSwindexSet;
            } else if (strcmp(thisParam, "y") == 0) {
                gbWantSaveSettings = true;
            } else if (strcmp(thisParam, "b") == 0) {
                gbBatchMode = true;
            } else if (strcmp(thisParam, "q") == 0) {
                gbQuietMode = true;
            } else {
                PrintHelp();
                return false;
            }
        } else {
            // Handle arguments that follow switches
            switch (state) {
                case wantDevID:
                    if (is_valid_ip(thisParam)) {
                        strncpy(gldadevicesip, thisParam, sizeof(gldadevicesip) - 1);
                        gldadevicesip[sizeof(gldadevicesip) - 1] = '\0';
                        gbDeviceOpen = true;
                    } else {
                        printf("Invalid IP Address\n");
                    }
                    state = wantDash;
                    break;

                case wantSwindex:
                    Swindex = atoi(thisParam);
                    gbWantSwitchStatus = true;
                    state = wantDash;
                    break;

                case wantSwindexSet:
                    Swindex = atoi(thisParam);
                    state = wantSWRFout;
                    break;

                case wantSWRFout:
                    Swout = atoi(thisParam);
                    gbWantSwitchCtrl = true;
                    state = wantDash;
                    break;

                default:
                    state = wantDash;
                    break;
            }
        }
    }

    if (state != wantDash) {
        // Expected another argument but didn’t get one
        PrintHelp();
        return false;
    }
    return true;
}

//***************************************************************************
// Main program
//*****************************************************************************
int main(int argc, char *argv[]) {
    bool realhardware = true;   // Set to false for test mode
    unsigned int index = 0;
    int swindex;

    LSWDEVICE_DATA_T lswdevicedata[1] = {0}; // Array for device info (single device)

    // Parse command line
    if (!ParseCommandLine(argc, argv))
        return 0;

    // Initialize API
    fnLSW_Init();
    fnLSW_SetTestMode(!realhardware);

    // Try to open the device
    if (gbDeviceOpen) {
        if (fnLSW_InitDevice(gldadevicesip))
            printf("Device Connection Error:%s\n", gldadevicesip);
        else
            printf("Device Connection Success:%s\n", gldadevicesip);
    }

    // ---------------------------------------------------------------------
    // Option: Read device parameters
    // ---------------------------------------------------------------------
    if (gbWantGetParam) {
        if (fnLSW_CheckDeviceReady(gldadevicesip) == STATUS_OK) {
            printf("Device IP:%s\n", gldadevicesip);
            printf("----------------------------------------------\n");

            fnLSW_GetModelName(gldadevicesip, lswdevicedata[index].modelname);
            printf("Model Name:%s\n", lswdevicedata[index].modelname);

            fnLSW_GetSerialNumber(gldadevicesip, &lswdevicedata[index].serialnumber);
            printf("Serial Number:%d\n", lswdevicedata[index].serialnumber);

            fnLSW_GetSoftwareVersion(gldadevicesip, lswdevicedata[index].swversion);
            printf("SW Version:%s\n", lswdevicedata[index].swversion);

            fnLSW_GetIPMode(gldadevicesip, &lswdevicedata[index].ipmode);
            printf("IP Mode:%d\n", lswdevicedata[index].ipmode);

            fnLSW_GetIPAddress(gldadevicesip, lswdevicedata[index].ipaddress);
            printf("IP Address:%s\n", lswdevicedata[index].ipaddress);

            fnLSW_GetNetmask(gldadevicesip, lswdevicedata[index].netmask);
            printf("Subnet Mask:%s\n", lswdevicedata[index].netmask);

            fnLSW_GetGateway(gldadevicesip, lswdevicedata[index].gateway);
            printf("Gateway:%s\n", lswdevicedata[index].gateway);

            fnLSW_GetMaxSwitchDevices(gldadevicesip, &lswdevicedata[index].maxswitchdevices);
            printf("Max Switch Devices:%d\n", lswdevicedata[index].maxswitchdevices);

            // Loop through all switches and print RF outputs
            for (swindex = 1; swindex <= lswdevicedata[index].maxswitchdevices; swindex++) {
                fnLSW_GetSwitchRFoutput(gldadevicesip, swindex,
                                        &lswdevicedata[index].rfswitchoutput[swindex - 1]);
                printf("SW%d-RF Output:%d\n", swindex,
                       lswdevicedata[index].rfswitchoutput[swindex - 1]);
            }
        }
        gbWantGetParam = false;
    }

    // ---------------------------------------------------------------------
    // Option: Get switch status
    // ---------------------------------------------------------------------
    if (gbWantSwitchStatus) {
        if (!gbQuietMode)
            printf("Read RF Switch Port status of switch%d for device %s\n",
                   Swindex, gldadevicesip);

        int resp = fnLSW_GetSwitchRFoutput(gldadevicesip, Swindex,
                                &lswdevicedata[index].rfswitchoutput[Swindex - 1]);
        if (resp == STATUS_OK) {
            printf("SW%d-RF Output:%d\n", Swindex,
                   lswdevicedata[index].rfswitchoutput[Swindex - 1]);
        } else {
            printf("Failed to get SW%d-RF Output\n", Swindex);
        }
    }

    // ---------------------------------------------------------------------
    // Option: Set switch port
    // ---------------------------------------------------------------------
    if (gbWantSwitchCtrl) {
        if (!gbQuietMode)
            printf("Set RF Switch Port output of switch%d to %d for device %s\n",
                   Swindex, Swout, gldadevicesip);
        int resp = fnLSW_SetSwitchRFoutput(gldadevicesip, Swindex, (LSW_SWPORT_T)Swout);
        if (resp == STATUS_OK) {
            printf("SW%d-RF Output set to %d\n", Swindex, Swout);
        } else {
            printf("Failed to set SW%d-RF Output\n", Swindex);
        }
    }

    // ---------------------------------------------------------------------
    // Option: Save settings
    // ---------------------------------------------------------------------
    if (gbWantSaveSettings) {
        if (!gbQuietMode)
            printf("Saving settings for device %s\n", gldadevicesip);
        fnLSW_SaveSettings(gldadevicesip);
    }

    // ---------------------------------------------------------------------
    // Close device connection at end
    // ---------------------------------------------------------------------
    if (gbDeviceOpen) {
        fnLSW_CloseDevice(gldadevicesip);
        gbDeviceOpen = false;

        // Clear out old device info
        memset(lswdevicedata, 0, sizeof(lswdevicedata));
    }

    return 0;
}
