"""
Vaunix RF Switch - Minimal Python Example using ctypes

What it demonstrates:
  1) Initialize library and (optionally) test mode
  2) Open device(s) by IP
  3) Read model, serial, network info
  4) Read RF output for each switch
  5) Set a switch output (1 -> 2 -> 1)
  6) Save settings
  7) Close device(s)

Adjust DEVICE_IPS below for your setup.
"""

import ctypes
from ctypes import c_int, c_bool, c_char_p, create_string_buffer
from time import sleep

# ---------------- Configuration ----------------
DEVICE_IPS = [b"192.168.1.100"]   # <<< change to your device IP(s)
LIB_PATH   = "./liblswsock.so"    # path to your shared library
REAL_HARDWARE = True              # set False to run in test mode

# Network buffer sizes your API expects
MAX_MODELNAME = 64
MAX_SWVERSION = 32
MAX_NETBUFF   = 16

# Status codes (adjust to match your API if different)
STATUS_OK = 0
STATUS_ERROR = 1

# ------------- Load library & set prototypes -------------
vnx = ctypes.CDLL(LIB_PATH)

# Basic init & housekeeping
vnx.fnLSW_Init.argtypes = []
vnx.fnLSW_Init.restype  = None

vnx.fnLSW_SetTestMode.argtypes = [c_bool]
vnx.fnLSW_SetTestMode.restype  = None

vnx.fnLSW_LibVersion.argtypes = []
vnx.fnLSW_LibVersion.restype  = c_char_p

# Device open/close & ready
vnx.fnLSW_InitDevice.argtypes = [c_char_p]
vnx.fnLSW_InitDevice.restype  = c_int

vnx.fnLSW_CloseDevice.argtypes = [c_char_p]
vnx.fnLSW_CloseDevice.restype  = c_int

vnx.fnLSW_CheckDeviceReady.argtypes = [c_char_p]
vnx.fnLSW_CheckDeviceReady.restype  = c_int

# Get basic info (model name into buffer, serial returned as int)
vnx.fnLSW_GetModelName.argtypes = [c_char_p, ctypes.c_char_p]
vnx.fnLSW_GetModelName.restype  = c_int  # returns number of chars (per your C example)

vnx.fnLSW_GetSerialNumber.argtypes = [c_char_p, ctypes.POINTER(c_int)]
vnx.fnLSW_GetSerialNumber.restype  = c_int  # simple int return in your socket API

# Optional: software version (into buffer)
vnx.fnLSW_GetSoftwareVersion.argtypes = [c_char_p, ctypes.c_char_p]
vnx.fnLSW_GetSoftwareVersion.restype  = c_int

# Network info
vnx.fnLSW_GetIPMode.argtypes = [c_char_p, ctypes.POINTER(c_int)]
vnx.fnLSW_GetIPMode.restype  = c_int

vnx.fnLSW_GetIPAddress.argtypes = [c_char_p, ctypes.c_char_p]
vnx.fnLSW_GetIPAddress.restype  = c_int

vnx.fnLSW_GetNetmask.argtypes = [c_char_p, ctypes.c_char_p]
vnx.fnLSW_GetNetmask.restype  = c_int

vnx.fnLSW_GetGateway.argtypes = [c_char_p, ctypes.c_char_p]
vnx.fnLSW_GetGateway.restype  = c_int

# Switch info & control (socket API returns values directly)
vnx.fnLSW_GetMaxSwitchDevices.argtypes = [c_char_p, ctypes.POINTER(c_int)]
vnx.fnLSW_GetMaxSwitchDevices.restype  = c_int

vnx.fnLSW_GetNumSwitches.argtypes = [c_char_p, ctypes.POINTER(c_int)]
vnx.fnLSW_GetNumSwitches.restype  = c_int

vnx.fnLSW_GetSwitchRFoutput.argtypes = [c_char_p, c_int, ctypes.POINTER(c_int)]
vnx.fnLSW_GetSwitchRFoutput.restype  = c_int

vnx.fnLSW_SetSwitchRFoutput.argtypes = [c_char_p, c_int, c_int]
vnx.fnLSW_SetSwitchRFoutput.restype  = c_int

# Save settings
vnx.fnLSW_SaveSettings.argtypes = [c_char_p]
vnx.fnLSW_SaveSettings.restype  = c_int

# ---------------- Helper functions ----------------
def perr(status: int) -> str:
    if status == STATUS_OK:
        return "STATUS_OK"
    elif status == STATUS_ERROR:
        return "STATUS_ERROR"
    return "UNKNOWN_STATUS"

def get_str(fn, dev_ip: bytes, buf_len: int) -> str:
    """Call a function that fills a char buffer (returns length or status)."""
    buf = create_string_buffer(buf_len)
    _ = fn(dev_ip, buf)
    return buf.value.decode("utf-8", errors="ignore")

# ---------------- Main demo flow ----------------
def main():
    # Init & mode
    vnx.fnLSW_Init()
    vnx.fnLSW_SetTestMode(not REAL_HARDWARE)

    libver = vnx.fnLSW_LibVersion().decode("utf-8")
    print(f"\nLSW Python demo (lib version: {libver})")
    print("---------------------------------------")

    if not DEVICE_IPS:
        print("No devices configured. Edit DEVICE_IPS and rerun.")
        return

    active = []

    # Open each device
    print("\nOpening devices...")
    for dev_ip in DEVICE_IPS:
        status = vnx.fnLSW_InitDevice(dev_ip)
        ok = (status == STATUS_OK)
        print(f"  {dev_ip.decode()}: open -> {status} ({perr(status)})")
        if ok:
            active.append(dev_ip)

    if not active:
        print("\nNo devices opened successfully. Exiting.")
        return

    # Work each active device
    for dev_ip in active:
        print(f"\n=== Device {dev_ip.decode()} ===")

        # Ready?
        ready = vnx.fnLSW_CheckDeviceReady(dev_ip)
        if ready != STATUS_OK:
            print(f"  Not ready: {ready} ({perr(ready)})")
            continue

        # Basic info
        model = get_str(vnx.fnLSW_GetModelName, dev_ip, MAX_MODELNAME)
        serial = c_int()
        vnx.fnLSW_GetSerialNumber(dev_ip, ctypes.byref(serial))
        swver = get_str(vnx.fnLSW_GetSoftwareVersion, dev_ip, MAX_SWVERSION)
        print(f"  Model        : {model}")
        print(f"  Serial Number: {serial.value}")
        print(f"  SW Version   : {swver}")

        # Network info
        ipmode = c_int()
        vnx.fnLSW_GetIPMode(dev_ip, ctypes.byref(ipmode))
        ipaddr = get_str(vnx.fnLSW_GetIPAddress, dev_ip, MAX_NETBUFF)
        netmask = get_str(vnx.fnLSW_GetNetmask, dev_ip, MAX_NETBUFF)
        gateway = get_str(vnx.fnLSW_GetGateway, dev_ip, MAX_NETBUFF)
        print(f"  IP Mode      : {ipmode.value}")
        print(f"  IP Address   : {ipaddr}")
        print(f"  Netmask      : {netmask}")
        print(f"  Gateway      : {gateway}")

        # Switch inventory
        max_switches = c_int()
        vnx.fnLSW_GetMaxSwitchDevices(dev_ip, ctypes.byref(max_switches))
        num_switches = c_int()
        vnx.fnLSW_GetNumSwitches(dev_ip, ctypes.byref(num_switches))
        print(f"  Max Switch Devices: {max_switches.value}")
        print(f"  Num Switches      : {num_switches.value}")

        if num_switches.value <= 0:
            print("  No switches reported by device.")
            continue

        # Read current RF outputs
        print("  Current RF outputs:")
        for i in range(1, max_switches.value + 1):
            rfout = c_int()
            vnx.fnLSW_GetSwitchRFoutput(dev_ip, i, ctypes.byref(rfout))
            print(f"    SW{i}: {rfout.value}")

        # Demo: Toggle switch 1 (if present) between ports 1 and 2
        swindex = 1
        if num_switches.value >= 1:
            print(f"\n  Toggling SW{swindex}...")
            for port in (1, 2, 1):
                status = vnx.fnLSW_SetSwitchRFoutput(dev_ip, swindex, port)
                if status == STATUS_OK:
                    print(f"    Set SW{swindex} -> {port} [OK]")
                else:
                    print(f"    Set SW{swindex} -> {port} [FAIL: {status} ({perr(status)})]")
                sleep(1.0)
                rfout = c_int()
                status = vnx.fnLSW_GetSwitchRFoutput(dev_ip, swindex, ctypes.byref(rfout))
                if status != STATUS_OK:
                    print(f"    Get SW{swindex} [FAIL: {status} ({perr(status)})]")
                print(f"    Read SW{swindex} -> {rfout.value}")
                sleep(0.5)

        if False:  # Optional: Save settings
            # Save settings
            st_save = vnx.fnLSW_SaveSettings(dev_ip)
            if st_save == STATUS_OK:
                print("  Saved settings.")
            else:
                print(f"  Save failed: {st_save} ({perr(st_save)})")

    # Close devices
    print("\nClosing devices...")
    for dev_ip in active:
        status = vnx.fnLSW_CloseDevice(dev_ip)
        print(f"  {dev_ip.decode()}: close -> {status} ({perr(status)})")

    print("\nDone.")

if __name__ == "__main__":
    main()
