// --------------------------------- lmsdrvr.h --------------------------------
//
//  Include file for Linux LabBrick LMS signal generator definitions
//
// (c) 2020-2024 by Vaunix Technology Corporation
//
//  JA  Version 1.0 LMS Digital Signal Generator Device Driver
//	RD	6/2024
//	RD	7/2024 updates for the BLX-403
//  NB  8/2025 updates for the BLX-403-20 and BLX-223
//
//-----------------------------------------------------------------------------
#ifndef __LMSDRVR_H__
#define __LMSDRVR_H__

//*****************************************************************************
//
// If building with a C++ compiler, make all of the definitions in this header
// have a C binding.
//
//*****************************************************************************
#ifdef __cplusplus
extern "C"
{
#endif

// ----------- Global Equates ------------
#define MAXDEVICES 			64
#define MAX_MODELNAME 		32
#define SEQ_MAX_ENTRIES		50

/// ---------- Macros ----------------
// Status returns for commands
#define LVSTATUS 			int
#define DEVID 				unsigned int

// ----------- Command Equates -----------
#define STOP_SEQUENCE		0			// stop a sequence
#define START_SEQUENCE		1			// start a sequence to run once
#define REPEAT_SEQUENCE		2			// start a repeating sequence
#define PAUSE_SEQUENCE		3			// pause a running sequence
#define RESUME_SEQUENCE		4			// resume a running sequence

// Status returns for commands
#define STATUS_OK 			0
#define BAD_PARAMETER 		0x80010000	// out of range input -- frequency outside min/max etc.
#define BAD_HID_IO 			0x80020000	// an error was returned by the lower level USB drivers
#define DEVICE_NOT_READY 	0x80030000	// device isn't open, no handle, etc.
#define ERROR_STATUS_MSK 	0x80000000	// the MSB is set for errors

// Status returns for EX commands that use unsigned 32 bit integers
// These values are above the representation for 42 GHz in 10 Hz units
#define STATUS_OK_EX 0
#define INVALID_DEVID_EX	0xFFFF0000	// if the device ID is invalid, 4,294,901,760 decimal, 
#define BAD_PARAMETER_EX	0xFFFF0001	// out of range input -- frequency outside min/max etc.
#define BAD_HID_IO_EX		0xFFFF0002
#define DEVICE_NOT_READY_EX	0xFFFF0003	// device isn't open, no handle, etc.

#define F_INVALID_DEVID 	-1.0		// for functions that return a float, STATUS_OK is the same
#define F_BAD_HID_IO 		-2.0
#define F_DEVICE_NOT_READY 	-3.0
#define F_BAD_PARAMETER		-4.0

// Status returns for DevStatus
#define INVALID_DEVID 		0x80000000 // MSB is set if the device ID is invalid
#define DEV_CONNECTED 		0x00000001 // LSB is set if a device is connected
#define DEV_OPENED 			0x00000002 // set if the device is opened
#define SWP_ACTIVE 			0x00000004 // set if the device is sweeping
#define SWP_UP 				0x00000008 // set if the device is sweeping up in frequency
#define SWP_REPEAT 			0x00000010 // set if the device is in continuous sweep mode
#define SWP_BIDIRECTIONAL 	0x00000020 // set if the device is in bi-directional sweep mode
#define PLL_LOCKED 			0x00000040 // set if the PLL lock status is TRUE (both PLL's are locked)
#define FAST_PULSE_OPTION 	0x00000080 // set if the fast pulse mode option is installed
#define EXT_SWEEP_OPTION 	0x00000100 // set if the external sweep option is enabled (HW may not be installed)

// Internal values in DevStatus
#define DEV_LOCKED 			0x00002000 // set if we don't want read thread updates of the device parameters
#define DEV_RDTHREAD 		0x00004000 // set when the read thread is running

// Flags to encode pulse mode time ranges
#define PM48Mhz 			0x10000000 // used to select the 48Mhz pulse mod clock
#define PM1Mhz 				0x00000000 // used to select the 1Mhz pulse mod clock or sw pulsing
#define PM120MHz			0x40000000 // used to select the 120Mhz pulse mode clock


// LMS Device Init
void fnLMS_Init(void);

// LMS Set Test mode
void fnLMS_SetTestMode(bool testmode);

// Get library version as an integer
int fnLMS_GetLibVersion(void);

// Get library version as a string
char *fnLMS_GetVersionStr(void);

// Get the number of LMS Devices present
int fnLMS_GetNumDevices();

// Get Information for the attached devices
int fnLMS_GetDevInfo(DEVID *ActiveDevices);

// Get LMS Device Model Name
int fnLMS_GetModelName(DEVID deviceID, char * ModelName);

// Init LMS Device to open a device for usage
int fnLMS_InitDevice(DEVID deviceID);

// LMS Device close connection
int fnLMS_CloseDevice(DEVID deviceID);

// Get Serial # of LMS Device
int fnLMS_GetSerialNumber(DEVID deviceID);

// Get Frequency of LMS Device
int fnLMS_GetFrequency(DEVID deviceID);
unsigned int fnLMS_GetFrequencyEx(DEVID deviceID);

// Get sweep Start Frequency of LMS Device
int fnLMS_GetStartFrequency(DEVID deviceID);
unsigned int fnLMS_GetStartFrequencyEx(DEVID deviceID);

// Get sweep End Frequency of LMS Device
int fnLMS_GetEndFrequency(DEVID deviceID);
unsigned int fnLMS_GetEndFrequencyEx(DEVID deviceID);

// Get freq Step size from the LMS Device
int fnLMS_GetFrequencyStep(DEVID deviceID);
unsigned int fnLMS_GetFrequencystepEx(DEVID deviceID);

// Get Sweep Time Configuration of LMS Device
int fnLMS_GetSweepTime(DEVID deviceID);

// Get Dwell Time for BLX sweep steps
int fnLMS_GetDwellTime(DEVID deviceID);

// Get Idle Time for BLX sweep steps
int fnLMS_GetIdleTime(DEVID deviceID);

// Get RF On status of LMS Device
int fnLMS_GetRF_On(DEVID deviceID);

// Get UseInternal Reference clock status
int fnLMS_GetUseInternalRef(DEVID deviceID);

// Get Use External Sweep Trigger status
int fnLMS_GetUseExtSweepTrigger(DEVID deviceID);

// Get Powerlevel Status of LMS Device
int fnLMS_GetPowerLevel(DEVID deviceID);

// Get Max Power Level of LMS Device
int fnLMS_GetMaxPwr(DEVID deviceID);

// Get Minimum Power Level of LMS Device
int fnLMS_GetMinPwr(DEVID deviceID);

// Get Maximum Frequency of LMS Device
int fnLMS_GetMaxFreq(DEVID deviceID);
unsigned int fnLMS_GetMaxFreqEx(DEVID deviceID);

// Get Minimum Frequency of LMS Device
int fnLMS_GetMinFreq(DEVID deviceID);
unsigned int fnLMS_GetMinFreqEx(DEVID deviceID);

// Get Pulse On Time of LMS Device
float fnLMS_GetPulseOnTime(DEVID deviceID);

// Get Pulse Off Time of LMS Device
float fnLMS_GetPulseOffTime(DEVID deviceID);

// Get Pulse Mode Status of LMS Device
int fnLMS_GetPulseMode(DEVID deviceID);

// Get Fast Pulse Mode status of device
int fnLMS_GetHasFastPulseMode(DEVID deviceID);

// Get Internal Pulse Mode Status of device
int fnLMS_GetUseInternalPulseMod(DEVID deviceID);

// Get sweep bi-directional mode of device
int fnLMS_GetSweepbidirectionalmode(DEVID deviceID);

// Get sweep mode of device
int fnLMS_GetSweepmode(DEVID deviceID);

// Get PLL lock status of device
int fnLMS_GetPLLLock(DEVID deviceID);

// Get the complete set of status bits
int fnLMS_GetDeviceStatus(DEVID deviceID);

// Chirp parameters
float fnLMS_GetChirpLength(DEVID deviceID);
float fnLMS_GetChirpRepeatRate(DEVID deviceID);
int fnLMS_GetHasChirpMode(DEVID deviceID);

// Set Frequency 
LVSTATUS fnLMS_SetFrequency(DEVID deviceID, int frequency);
LVSTATUS fnLMS_SetFrequencyEx(DEVID deviceID, unsigned int frequency);

// Set sweep Start Frequency
LVSTATUS fnLMS_SetStartFrequency(DEVID deviceID, int startfrequency);
LVSTATUS fnLMS_SetStartFrequencyEx(DEVID deviceID, unsigned int startfrequency);

// Set sweep End Frequency
LVSTATUS fnLMS_SetEndFrequency(DEVID deviceID, int endfrequency);
LVSTATUS fnLMS_SetEndFrequencyEx(DEVID deviceID, unsigned int endfrequency);

// Set Frequency Step 
LVSTATUS fnLMS_SetFrequencystep(DEVID deviceID, int freqstep);

// Set dwell time for BLX sweep steps
LVSTATUS fnLMS_SetDwellTime(DEVID deviceID, int dwelltime);

// Set idle time for BLX sweep
LVSTATUS fnLMS_SetIdleTime(DEVID deviceID, int idletime);

// Set Power Level 
LVSTATUS fnLMS_SetPowerLevel(DEVID deviceID, int powerlevel);

// Set RF On Status
LVSTATUS fnLMS_SetRFOn(DEVID deviceID, bool on);

// Set Pulse On Time
LVSTATUS fnLMS_SetPulseOnTime(DEVID deviceID, float pulseontime);

// Set Pulse Off Time
LVSTATUS fnLMS_SetPulseOffTime(DEVID deviceID, float pulseofftime);

// Set Fast Pulsed Output mode
LVSTATUS fnLMS_SetFastPulsedOutput(DEVID deviceID, float pulseontime, float pulsereptime, bool on);

// Enable Pulse Modulation
LVSTATUS fnLMS_EnableInternalPulseMod(DEVID deviceID, bool on);

// Set External Pulse Mode
LVSTATUS fnLMS_SetUseExternalPulseMod(DEVID deviceID, bool external);

// Set Internal Reference
LVSTATUS fnLMS_SetUseInternalRef(DEVID deviceID, bool internal);

// Set External sweep trigger
LVSTATUS fnLMS_SetUseExtSweepTrigger(DEVID deviceID, bool external);

// Set Sweep time
LVSTATUS fnLMS_SetSweepTime(DEVID deviceID, int sweeptime);

// Set Sweep Direction
LVSTATUS fnLMS_SetSweepDirection(DEVID deviceID, bool up);

// Set Sweep bidirectional
LVSTATUS fnLMS_SetSweepbidirection(DEVID deviceID, bool bidir);

// Set Sweep Mode
LVSTATUS fnLMS_SetSweepMode(DEVID deviceID, bool mode);

// Set Sweep Type
LVSTATUS fnLMS_SetSweepType(DEVID deviceID, bool swptype);

// Start Sweep
LVSTATUS fnLMS_StartSweep(DEVID deviceID, bool go);

// Chirp parameters
LVSTATUS fnLMS_SetChirpMode(DEVID deviceID, bool mode);
LVSTATUS fnLMS_SetChirpLength(DEVID deviceID, float chirplength);
LVSTATUS fnLMS_SetChirpRepeatRate(DEVID deviceID, float chirp_repeat);

// Sequence parameters
LVSTATUS fnLMS_SetSequenceElement(DEVID deviceID, int index, unsigned int frequency, int powerlevel, bool pwr_control);
LVSTATUS fnLMS_SetSequenceStart(DEVID deviceID, int start);
LVSTATUS fnLMS_SetSequenceCount(DEVID deviceID, int count);
LVSTATUS fnLMS_SetSequenceDwellTime(DEVID deviceID, int dwelltime);
LVSTATUS fnLMS_SetSequenceIdleTime(DEVID deviceID, int idletime);
LVSTATUS fnLMS_StartSequence(DEVID deviceID, int control);

// Save Settings for persistence 
LVSTATUS fnLMS_SaveSettings(DEVID deviceID);

// Error Status
char *fnLMS_perror(LVSTATUS status);
char *fnLMS_pFloatError(float fstatus);

//*****************************************************************************
//
// Mark the end of the C bindings section for C++ compilers.
//
//*****************************************************************************
#ifdef __cplusplus
}
#endif

#endif // __LMSDRVR_H__                             
