// -------------------------------------------------------------------------------------------------------------------------
// Module Name: LDA Ethernet Socket Communication Device Control Linux Library
// JA 01/29/2020    Initial Verison oF Ethernet Library
// RD 05/03/2024    Updates, fixes, new mc functions etc. -------------------------------------------------------------------------------------------------------------------------
// This library uses .05db units for attenuation values, so 10db is represented by 200. (multiply by 20 to convert db to api units)
/// ---------- Include headers ----------------
#include <stdbool.h>
#include <stdint.h>         
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <time.h>
#include <errno.h>
#include <sys/select.h>
#include <sys/socket.h>
#include <arpa/inet.h> 
#include <netdb.h>
#include <poll.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <errno.h>
#include <netinet/in.h>
#include <sys/timerfd.h>
#include <sys/wait.h> 
#include "blxsocket.h"

/// ---------- Macros ----------------
#define _CRT_SECURE_CPP_OVERLOAD_SECURE_NAMES 	1
#define _CRT_SECURE_NO_WARNINGS					1

#define BLX_SOCKET_PORT  "40001"
#define FALSE 0
#define TRUE 1          
#define PACKET_CTRL_LEN 8
#define PACKET_INT_LEN  8

#define LIBVER "1.0.0"
#define LMS_DLLVERSION 0x100        // we return an integer representation of the version with one byte each for major and minor version

// to force LMS library debugging messages choose the following definitions
#define DEBUG_OUT 0     /* set this to 1 in order to see debugging output, 2 for a ton of output, or 3 for many tons */

// IP Address Validator
#define DELIM "."

// Socket connetion timeout
#define CONNECTION_TIMEOUT_SEC 5

// Command Send Throttling Macros
#define TCP_SEND_INTERVAL 200       // This is the default delayed ACK time
#define MAX_COMMANDS_PER_WINDOW 48

// Thread States Macros
#define THREAD_IDLE 0
#define THREAD_START 1
#define THREAD_EXIT 3
#define THREAD_DEAD 4
#define THREAD_ERROR -1

// --------------------- other device specific equates ------------------------
#define HW_MAXP 40			// MaxPower is the output of the device in db -- +10 in this case
                            // NB -- the value used in the SetPower function is relative attenuation
                            // not absolute power!!

#define HW_MAXP_20 80		// MaxPower for the +20 dbm devices


/* -----------------------------------------------------------------
Global Data to maintain the Devices data & states for each process
-------------------------------------------------------------------*/
bool TestMode = FALSE;          // if TestMode is true we fake it -- no HW access
int  lmsdevcount = 0;
time_t starttime;
LMSPARAMS lms[MAXDEVICES];      // an array of structures each of which holds the info for a given device

/* stuff for throttling */
pthread_mutex_t throttle_locks[MAXDEVICES] = { [0 ... MAXDEVICES-1] = PTHREAD_MUTEX_INITIALIZER };
int win_command_counts[MAXDEVICES] = {0};

static struct timespec prev_cmd_times[MAXDEVICES] = {0};

/* stuff for the threads */
pthread_t threads[MAXDEVICES];


/// ---------- Const Defintions----------
const char* sVNX_devicenames[] = { "BLX-403", "BLX-403-20", "BLX-223" };

const char STATUS_QUERY_ID[]= { VNX_SETSERNUM, VNX_FREQUENCY, VNX_FDWELL, VNX_FSTART, VNX_FSTOP, VNX_FSTEP, VNX_SWEEP,
                         VNX_RFMUTE, VNX_ATTEN, VNX_PWR, VNX_DSS_STATUS, VNX_INTOSC, VNX_AIDLE, VNX_EXTSWP, VNX_SETPROFILE, 
                         VNX_PROFILECOUNT, VNX_PROFILEDWELL, VNX_PROFILEIDLE, VNX_PROFILESTART, VNX_OFFTIME, VNX_ONTIME, 
                         VNX_PULSE_MODE, VNX_MAX_PWR, VNX_MINFREQUENCY, VNX_MAXFREQUENCY, VNX_MODELNAME, VNX_IPMODE, 
                         VNX_IPADDR, VNX_IPMASK, VNX_IPGW };

char LibVersion[] = LIBVER;

// Forward declarations
bool GetSequenceElement(DEVID deviceID, int index);
int WF_GetSequenceElement(DEVID deviceID, int index);
static int vnxDeviceOpen(char* deviceip);
bool GetParameter(DEVID deviceID, int GetParam);
bool RawSend(int deviceID, unsigned char command, unsigned char *pBuffer, int cbBuffer);

static void *rxdatahandler_callback(void *threadID);

/// ----------  Static Fucntion Calls ----------------
//***************************************************************************
//
// Function call for returning the LibVersion
//
//*****************************************************************************
char* fnLMS_LibVersion(void) {
    return LibVersion;
}

//***************************************************************************
//
// Function call for returning the DLL Library Version
//
//*****************************************************************************
int fnLMS_GetLibVersion(void){
    return LMS_DLLVERSION;
}

//***************************************************************************
//
// Function call for Settig into test mode
//
//*****************************************************************************
void fnLMS_SetTestMode(bool testmode) {
    TestMode = testmode;
}

//***************************************************************************
//
// Function call for delay loop
//
//*****************************************************************************
/* usleep is deprecated, so this is a function using nanosleep() */
void sleep_ms(long naptime) {
  // naptime comes in as the number of ms we want. 20 = 20ms
  struct timespec timeval;
   timeval.tv_sec = 0;
   timeval.tv_nsec = naptime * 1000000L;

   nanosleep(&timeval , NULL);
}

//***************************************************************************
//
// Wait for the device's buffer to be empty
//
//*****************************************************************************
bool wait_for_empty_buffer(int devid) {
    const int MAX_RETRIES = 10;
    int retries = 0;
    BYTE VNX_param[6] = {0, 0, 0, 0, 0, 0};

    RawSend(devid, VNX_DSS_STATUS, VNX_param, 0);
    while ((lms[devid].DevStatus & DEV_BUFEMPTY) == 0 && retries++ < MAX_RETRIES) {
        sleep_ms(TCP_SEND_INTERVAL);
        RawSend(devid, VNX_DSS_STATUS, VNX_param, 0);
    }

    if (retries >= MAX_RETRIES) {
        if (DEBUG_OUT > 0) {
            printf("Warning: Timeout waiting for buffers to empty on device %d\n", devid);
            return false;
        }
    }

    return true;
}

//***************************************************************************
//
// Throttle the number of commands sent to the device if we are sending too many
//
//****************************************************************************
bool maybe_throttle(int devid) {
    int cur_ms;
    int prev_ms;
    struct timespec ts;

    // Get the current time
    clock_gettime(CLOCK_MONOTONIC, &ts);
    // Calculate the time in ms of the start of the window versus current time in ms
    prev_ms = (prev_cmd_times[devid].tv_sec * 1000) + (prev_cmd_times[devid].tv_nsec / 1000000);
    cur_ms = (ts.tv_sec * 1000) + (ts.tv_nsec / 1000000);

    if (cur_ms - prev_ms > TCP_SEND_INTERVAL) {
        // Reset the window
        win_command_counts[devid] = 0;
    } else if (win_command_counts[devid] > MAX_COMMANDS_PER_WINDOW && prev_ms != 0) {
        // The maximum command count has been hit for this window
        sleep_ms(TCP_SEND_INTERVAL);
        // Check if the buffer is empty
        if (!wait_for_empty_buffer(devid)) return false;

        win_command_counts[devid] = 0;
    }

    // Keep track of the time for this command send
    clock_gettime(CLOCK_MONOTONIC, &ts);
    prev_cmd_times[devid] = ts;
    
    // Increment the command counter
    win_command_counts[devid]++;
    return true;
}

//***************************************************************************
//
// Validate netconfig digits and return 1 if string contain only digits, else return 0
//
//*****************************************************************************
int valid_digit(char *ip_str){
    while (*ip_str) {
        if (*ip_str >= '0' && *ip_str <= '9')
            ++ip_str;
        else
            return 0;
    }
    return 1;
}

//***************************************************************************
//
// Validate IP Configuration -return 1 if IP string is valid, else return 0
//
//*****************************************************************************
int is_valid_ip(char *ip_str){
    int num, dots = 0;
    char *ptr;
    char lstr[16];
    char* next_token;
    strcpy(lstr, ip_str);

    if (lstr == NULL)
        return 0;

    ptr = strtok(lstr, DELIM);

    if (ptr == NULL)
        return 0;

    while (ptr) {

        /* after parsing string, it must contain only digits */
        if (!valid_digit(ptr))
            return 0;

        num = atoi(ptr);

        /* check for valid IP */
        if (num >= 0 && num <= 255) {
            /* parse remaining string */
            ptr = strtok(NULL, DELIM);
            if (ptr != NULL)
                ++dots;
        } else
            return 0;
    }

    /* valid IP string must contain 3 dots */
    if (dots != 3)
        return 0;
    return 1;
}

//***************************************************************************
//
// Function call to Get DeviceID based on the deviceip from the device entry list
//
//*****************************************************************************
int GetDeviceID(char* deviceip){
    int index=-1;

    // Loop through to get the device index
    for(index=0; index < MAXDEVICES; index++)
    {
        if(strcmp(lms[index].deviceip, deviceip)==0)
            return index;
    }

    return index;
}

//***************************************************************************
//
// Function call to check whether LMS device socket open or not
//
//*****************************************************************************
bool isDeviceSockOpen(DEVID deviceID) {
    if (TestMode) return TRUE;// in test mode all devices are always available

    //  printf("devid:%d\n",devid);
    if(deviceID >=0)
    {
        //      printf("devicesockfd:%d\n",lms[devid].devicesockfd);
        if (lms[deviceID].devicesockfd != 0)
            return TRUE;
        else
            return FALSE;
    }
    else
        return FALSE;
}

//***************************************************************************
//
// Function check if the device is locked or not
//
//*****************************************************************************
bool DevNotLocked(DEVID deviceID)
{
    if (TestMode) return true;        // this shouldn't happen, but just in case...

    if (!(lms[deviceID].DevStatus & DEV_LOCKED))
    {
        return true;                // we return true if the device is not locked!
    }
    else return false;
}

//***************************************************************************
//
// Function call to lock the device
//
//*****************************************************************************
void LockDev(DEVID deviceID, bool lock)
{
    if (TestMode) return;        // this shouldn't happen, but just in case...

    if (lock)
    {
        lms[deviceID].DevStatus = lms[deviceID].DevStatus | DEV_LOCKED;
        return;
    }
    else
    {
        lms[deviceID].DevStatus = lms[deviceID].DevStatus & ~DEV_LOCKED;
        return;
    }
}

//***************************************************************************
//
// -- our main function to find LMS attached devices --
//
// RD test version, it just counts the number of devices that have an IP address
// later versions will open a socket to the device and get a serial number and name
// for each device. (This is needed to support the LMS API functions which get serial numbers
// and model information for devices that are not opened yet)
//
// Later versions will set the DEV_CONNECTED flag, and gather the basic parameters
//
//***************************************************************************
int FindVNXDevices(void)
{
    int i;
    int j;
    int NumDevices = 0;             // we will total up how many LMS devices we find
    int sockfd;
    int err;
    int validModel = 0;

    for (i = 1; i < MAXDEVICES; i++)
    {
        if (is_valid_ip(lms[i].deviceip)) {
            // Try to open a socket to the device
            sockfd = vnxDeviceOpen(lms[i].deviceip);
            if (sockfd > 0) {
                lms[i].devicesockfd = sockfd;
                
                // Create a device threads for reading the data from the device using sockets
                err = pthread_create(&threads[i], NULL, rxdatahandler_callback, (void*)(uintptr_t)i);

                // Query serial number and model name
                if (GetParameter(i, VNX_SETSERNUM) && GetParameter(i, VNX_MODELNAME) &&
                    GetParameter(i, VNX_MAXFREQUENCY) && GetParameter(i, VNX_MINFREQUENCY)) {
                    // Loop through each model name to check if this device is one we want
                    for (int j = 0; j < (sizeof(sVNX_devicenames) / sizeof(sVNX_devicenames[0])); j++) {
                        if (strcmp(lms[i].ModelName, sVNX_devicenames[j]) == 0) {
                            // We found one of our devices
                            validModel = 1;
                            break;
                        }
                    }
                    if (validModel) {
                        lms[i].DevStatus |= DEV_CONNECTED;
                        lms[i].DevStatus |= DEV_OPENED;
                        NumDevices++;
                    }
                } else {
                    lms[i].DevStatus &= ~DEV_CONNECTED;
                }
            } else {
                lms[i].DevStatus &= ~DEV_CONNECTED;
            }
            
            // Close down the socket connection
            close(sockfd);
            lms[i].devicesockfd = 0;
            lms[i].DevStatus &= ~DEV_OPENED;    // Remove the dev opened flag
        } else {
            lms[i].DevStatus &= ~DEV_CONNECTED;
        }
    }

    return NumDevices;
}

//***************************************************************************
//
// Function call to check whether lms device socket open or not
//
//*****************************************************************************
static int vnxDeviceOpen(char* deviceip) {
    int sockfd = -1;
    struct addrinfo hints;
    struct addrinfo *result, *rp;
    int flags;
    int iResult;

    memset(&hints, 0, sizeof(struct addrinfo));
    hints.ai_canonname = NULL;
    hints.ai_addr = NULL;
    hints.ai_next = NULL;
    hints.ai_family = AF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    hints.ai_flags = AI_NUMERICSERV;

    if (DEBUG_OUT > 1) printf("IP:%s\n",deviceip);
 
    // Convert IPv4 and IPv6 addresses from text to binary form 
    if(inet_pton(AF_INET, deviceip, &hints.ai_addr)<=0)  
    { 
        if (DEBUG_OUT > 0) printf("\nInvalid address/ Address not supported \n"); 
        return DEVICE_NOT_READY; 
    } 

    // Check if Device is already open or not
    if (getaddrinfo(deviceip, BLX_SOCKET_PORT, &hints, &result) != 0)
        return DEVICE_NOT_READY;

    for (rp = result; rp != NULL; rp = rp->ai_next)
    {
        //  Open the Socket Connection 
        sockfd = socket(rp->ai_family, rp->ai_socktype, rp->ai_protocol);
        if (sockfd == -1)
            continue;

        // Set socket to non-blocking
        flags = fcntl(sockfd, F_GETFL, 0);
        if (flags == -1) {
            close(sockfd);
            continue;
        }
        if (fcntl(sockfd, F_SETFL, flags | O_NONBLOCK) == -1) {
            close(sockfd);
            continue;
        }

        iResult = connect(sockfd, rp->ai_addr, rp->ai_addrlen);
        if (iResult < 0) {
            if (errno != EINPROGRESS) {
                close(sockfd);
                sockfd = -1;
                continue;
            }
        }

        if (iResult < 0) {
            fd_set writefds;
            FD_ZERO(&writefds);
            FD_SET(sockfd, &writefds);

            struct timeval tv;
            tv.tv_sec = CONNECTION_TIMEOUT_SEC;
            tv.tv_usec = 0;

            iResult = select(sockfd + 1, NULL, &writefds, NULL, &tv);
            if (iResult <= 0) {
                close(sockfd);
                sockfd = -1;
                continue;
            }

            // Check if the connection actually succeeded
            int optVal;
            socklen_t optLen = sizeof(optVal);
            if (getsockopt(sockfd, SOL_SOCKET, SO_ERROR, &optVal, &optLen) < 0 || optVal != 0) {
                close(sockfd);
                sockfd = -1;
                continue;
            }
        }

        // Restore blocking mode
        if (fcntl(sockfd, F_SETFL, flags & ~O_NONBLOCK) == -1) {
            close(sockfd);
            sockfd = -1;
            continue;
        }

        //Successfully connected
        break;
    }

    if (rp == NULL)
    {
        if (sockfd != -1) close(sockfd);
        return DEVICE_NOT_READY;
    }

    freeaddrinfo(result);

    return sockfd;
}

//***************************************************************************
//
// Function call to Send the VNX commands through the Socket to the given ip
//  device
//*****************************************************************************
bool SendReport(int deviceID, unsigned char command, unsigned char *pBuffer, int cbBuffer)
{
    int index;
    int timedout;

    // Check if we are sending too many commands to the device
    // This will wait a little bit if so
    pthread_mutex_lock(&throttle_locks[deviceID]);
    if (!maybe_throttle(deviceID)) {
        // Something went wrong with throttling, don't send!
        pthread_mutex_unlock(&throttle_locks[deviceID]);
        return FALSE;
    }
    pthread_mutex_unlock(&throttle_locks[deviceID]);

    // Make sure the buffer that is being passed to us fits
    if (cbBuffer > HR_BLOCKSIZE) {
        // Report too big, don't send!
        return FALSE;
    }

    // lets make sure our command doesn't have any junk bits in it
    for (index=0; index<HID_REPORT_LENGTH; index++)
        lms[deviceID].sndbuff[index] = 0;

    if (DEBUG_OUT > 1) printf("SR: command=%x cbBuffer=%d\r\n", (uint8_t)command, cbBuffer);
    lms[deviceID].sndbuff[0] = command;       // command to device
    lms[deviceID].sndbuff[1] = cbBuffer;
    for (index=0; index<cbBuffer; index++)
        lms[deviceID].sndbuff[index+2] = pBuffer[index];

    if (DEBUG_OUT > 1) {
    printf("SR to device %d: ", deviceID);
    for (index=0; index<8; index++) {
        printf("%02x ", (uint8_t)lms[deviceID].sndbuff[index]);
    }
    printf("\r\n");
    }

    lms[deviceID].responseready = 0; // Clear Device Response before command send

    // Call Socket Function to send packet data to the device
    if (write(lms[deviceID].devicesockfd, lms[deviceID].sndbuff, PACKET_CTRL_LEN) < 0 )
        if (DEBUG_OUT > 0) printf("sending Message Error!");

    //Check for Get /Set Message type and expect response if Get message only
    if((command &  VNX_SET) == 0)
    {
        // --- then we wait for the read thread's parser to signal that it got the response
        starttime = time(NULL);
        timedout = 0;
        
        // wait until the value is decoded or 1 seconds have gone by
        // RD 4-2019 modified to yield the thread during the wait
        while ((!lms[deviceID].responseready) && (0 == timedout)) {
        sleep_ms(10);   // yield for 10 ms, speed matters here
        if ((time(NULL)-starttime) > 1) {
            if (DEBUG_OUT > 0) printf("Timeout waiting for response to %x command", command);
            timedout = 1;
        }
        }

        return (0 == timedout) ? TRUE : FALSE;
    }
    else
        return TRUE;
}

//***************************************************************************
//
// Minimal, fast send with no throttling
//
//*****************************************************************************
bool RawSend(int deviceID, unsigned char command, unsigned char *pBuffer, int cbBuffer)
{
    int index;
    int timedout;

    // Make sure the buffer that is being passed to us fits
    if (cbBuffer > HR_BLOCKSIZE) {
        // Report too big, don't send!
        return FALSE;
    }

    // lets make sure our command doesn't have any junk bits in it
    for (index=0; index<HID_REPORT_LENGTH; index++)
        lms[deviceID].sndbuff[index] = 0;

    if (DEBUG_OUT > 1) printf("SR: command=%x cbBuffer=%d\r\n", (uint8_t)command, cbBuffer);
    lms[deviceID].sndbuff[0] = command;       // command to device
    lms[deviceID].sndbuff[1] = cbBuffer;
    for (index=0; index<cbBuffer; index++)
        lms[deviceID].sndbuff[index+2] = pBuffer[index];

    if (DEBUG_OUT > 1) {
    printf("SR to device %d: ", deviceID);
    for (index=0; index<8; index++) {
        printf("%02x ", (uint8_t)lms[deviceID].sndbuff[index]);
    }
    printf("\r\n");
    }

    lms[deviceID].responseready = 0; // Clear Device Response before command send

    // Call Socket Function to send packet data to the device
    if (write(lms[deviceID].devicesockfd, lms[deviceID].sndbuff, PACKET_CTRL_LEN) < 0 )
        if (DEBUG_OUT > 0) printf("sending Message Error!");

    //Check for Get /Set Message type and expect response if Get message only
    if((command &  VNX_SET) == 0)
    {
        // --- then we wait for the read thread's parser to signal that it got the response
        starttime = time(NULL);
        timedout = 0;
        
        // wait until the value is decoded or 1 seconds have gone by
        // RD 4-2019 modified to yield the thread during the wait
        while ((!lms[deviceID].responseready) && (0 == timedout)) {
        sleep_ms(10);   // yield for 10 ms, speed matters here
        if ((time(NULL)-starttime) > 1) {
            if (DEBUG_OUT > 0) printf("Timeout waiting for response to %x command", command);
            timedout = 1;
        }
        }

        return (0 == timedout) ? TRUE : FALSE;
    }
    else
        return TRUE;
}

//***************************************************************************
//
// Function call to clear Cache for the device data structures with default data
//  
//*****************************************************************************
void ClearDevCache(DEVID deviceID) {
    unsigned int index;

    lms[deviceID].Frequency = 0;           // we mark the attenuation as unknown to start
    lms[deviceID].StartFrequency = 0;      // and everybody else...
    lms[deviceID].EndFrequency = 0;
    lms[deviceID].FrequencyStep = 0;
    lms[deviceID].DwellTime = -1;
    lms[deviceID].SweepIdle = -1;
    lms[deviceID].Power = -1;
    lms[deviceID].MaxPower = -1;
    lms[deviceID].MinPower = -1;
    lms[deviceID].MaxAttenuation = -1;
    lms[deviceID].PulseOnTime = -1;
    lms[deviceID].PulseOffTimeSeconds = -1;
    lms[deviceID].HasOldBLX_FW = FALSE;
    lms[deviceID].Modebits = 0;               // Modebits aren't directly readable, but clearing them here for convenience
    lms[deviceID].SequenceLength = 0;
    lms[deviceID].SequenceStart = 0;
    lms[deviceID].SequenceDwell = 0;
    lms[deviceID].SequenceIdle = 0;
    lms[deviceID].SequenceMode = 0;
    lms[deviceID].ProfileActive = FALSE;
    
    // clear the profile cache
    for (index = 0; index < lms[deviceID].SequenceMaxLength; index++) {
        lms[deviceID].Sequence[index].Frequency10Hz = BLX_SEQ_EMPTY;     // set all the profile elements to PROFILE_EMPTY
    }

}


/*******************************************************************************
 * \brief    Parse Data Frame received from the socket
 *
 ******************************************************************************/
int parseDataFrame(char* msg, int count, int sockfd, int devid)
{
    int i;
    RESPONSE_DATA_T dataresp;
    unsigned short dataval_16;
    unsigned int dataval_32;
    unsigned char dataval_8;
    char *dataptr;
    char namebuffer[7];

#if 0
    int index;
    for(index=0; index < count; index++)
    {
        printf("%02X",(unsigned char)msg[index]);
    }

    printf("Parse Data Frame:(Sockfd-%d, Devid-%d, Count-%d)\n",sockfd,devid,count);
#endif

    dataresp.command = msg[0]; // Command ID
    dataresp.length = msg[1];  // Data Length
    dataptr = &msg[2]; // Data buffer Start
    dataval_16 = *(unsigned short*)&msg[2];   // Data of 16 bits
    dataval_32 = *(unsigned int*)&msg[2]; // Data of 32 bits
    dataval_8 =  *(unsigned char*)&msg[2]; // Data of 8bits

    // handle the status report
    if (dataresp.command == VNX_DSS_STATUS) {

        if (DevNotLocked(devid))
        {
            // Check if buffers are empty (msg[1] is the length + status byte, we need to check the two highest bits)
            if ((msg[1] & 0xC0) == 0) {
                lms[devid].DevStatus = lms[devid].DevStatus | DEV_BUFEMPTY;
                if (DEBUG_OUT > 2) printf("Device %d: Buffers are empty\n", devid);
            } else {
                lms[devid].DevStatus = lms[devid].DevStatus & ~DEV_BUFEMPTY;
                if (DEBUG_OUT > 2) printf("Device %d: Buffers are NOT empty\n", devid);
            }

            // for the BLX-403 some early versions reported in 100Hz units. We will fix up those reports once we know that they
            // are coming from old firmware. Until then we just do our best.
            // We used to use the serial number:
            // if (lms[MyDev].DevType == HW_BLX403 && lms[MyDev].SerialNumber < 32915)		// Scott says Stellant HW was 32205 to 32914

            if (lms[devid].DevType == HW_BLX403)
            {
                if (lms[devid].HasOldBLX_FW)
                {
                    // we do a sanity check first
                    if (dataval_32 >= 5000000U && dataval_32 <= 400000000U)
                    {
                        lms[devid].Frequency = dataval_32 * 10;	// update the frequency, working around wrong units for early production!
                    }
                    else
                    {
                        if (DEBUG_OUT > 3)printf(" Invalid BLX Legacy Frequency = %d \n", dataval_32);
                    }
                }
                else
                {
                    // we are probably running the new firmware, or we got this report before the firmware level was determined
                    // we do a sanity check first
                    if (dataval_32 >= 50000000U && dataval_32 <= 4000000000U)
                    {
                        lms[devid].Frequency = dataval_32;	// update the frequency
                    }
                    else
                    {
                        if (DEBUG_OUT > 3)printf(" Invalid BLX Frequency = %d \n", dataval_32);
                    }
                }
            }
            else
            {
                lms[devid].Frequency = dataval_32;		// update the frequency - other devices always send the correct 10Hz units from HW
            }

            if (msg[6] & (STATUS_PLL_LOCK))		// is the PLL locked?
            {
                lms[devid].DevStatus = lms[devid].DevStatus | PLL_LOCKED;
            }
            else
            {
                lms[devid].DevStatus = lms[devid].DevStatus & ~PLL_LOCKED;
            }

            // -- update the sweep status bits --
            if (msg[6] & (SWP_ONCE | SWP_CONTINUOUS))		// are we sweeping?
            {
                lms[devid].DevStatus = lms[devid].DevStatus | SWP_ACTIVE;
            }
            else
            {
                lms[devid].DevStatus = lms[devid].DevStatus & ~SWP_ACTIVE;
            }

            // -- fill in the SWP_UP status bit
            if (msg[6] & (SWP_DIRECTION))					// are we sweeping downwards?
            {
                lms[devid].DevStatus = lms[devid].DevStatus & ~SWP_UP;
            }
            else
            {
                lms[devid].DevStatus = lms[devid].DevStatus | SWP_UP;
            }

            // -- fill in the continuous sweep bit
            if (msg[6] & (SWP_CONTINUOUS))					// are we in continuous sweep mode
            {
                lms[devid].DevStatus = lms[devid].DevStatus | SWP_REPEAT;
            }
            else
            {
                lms[devid].DevStatus = lms[devid].DevStatus & ~SWP_REPEAT;
            }

            // -- fill in the sweep type bit
            if (msg[6] & (SWP_BIDIR))						// are we in bi-directional sweep mode
            {
                lms[devid].DevStatus = lms[devid].DevStatus | SWP_BIDIRECTIONAL;
            }
            else
            {
                lms[devid].DevStatus = lms[devid].DevStatus & ~SWP_BIDIRECTIONAL;
            }
        }
    }

    switch(dataresp.command){

        case VNX_MODELNAME:
            if (DevNotLocked(devid))
            {
                // build a string from the 6 name bytes we received
                for(i = 0; i < 6; i++)
                {
                    namebuffer[i] = msg[2+i];
                }
                namebuffer[6] = 0;

                if (strstr(namebuffer,"03-20")) {
                    strcpy(lms[devid].ModelName, "BLX-403-20");     // Fix model name of "03-20"
                    lms[devid].DevType = HW_BLX403_20;
                    lms[devid].MinFrequency = 50000000;
                    lms[devid].MaxFrequency = 4000000000;
                    lms[devid].MaxAttenuation = 200;        // 50db for the BLX-403-20 (-30db with max = +20)
                    lms[devid].MaxPower = HW_MAXP_20;
                    lms[devid].MinPower = lms[devid].MaxPower - lms[devid].MaxAttenuation;
                    lms[devid].SequenceMaxLength = SEQ_MAX_ENTRIES;
                }
                if (strstr(namebuffer,"-403")) {
                    strcpy(lms[devid].ModelName, "BLX-403");     // Fix model name of "X-403"
                    lms[devid].DevType = HW_BLX403;
                    lms[devid].MinFrequency = 50000000;
                    lms[devid].MaxFrequency = 4000000000;
                    lms[devid].MaxAttenuation = 180;        // 45db for the BLX-403 (-35db with max = +10)
                    lms[devid].MaxPower = HW_MAXP;
                    lms[devid].MinPower = lms[devid].MaxPower - lms[devid].MaxAttenuation;
                    lms[devid].SequenceMaxLength = SEQ_MAX_ENTRIES;
                }
                else if (strstr(namebuffer,"-223")) {
                    strcpy(lms[devid].ModelName, "BLX-223");     // Fix model name of "X-223"
                    lms[devid].DevType = HW_BLX223;
                    lms[devid].MinFrequency = 70000000;
                    lms[devid].MaxFrequency = 2200000000;
                    lms[devid].MaxAttenuation = 160;        // 40db for the BLX-223 (-30db with max = +10)
                    lms[devid].MaxPower = HW_MAXP;
                    lms[devid].MinPower = lms[devid].MaxPower - lms[devid].MaxAttenuation;
                    lms[devid].SequenceMaxLength = SEQ_MAX_ENTRIES;
                }
            }
            break;

        case VNX_IPMODE:
            if (DevNotLocked(devid))
            {
                lms[devid].ipmode = (int)dataval_8;
            }
            break;

        case VNX_IPADDR:
            if (DevNotLocked(devid))
            {
                sprintf(lms[devid].ipaddress, "%d.%d.%d.%d", (dataval_32 >> 24) & 0xff, (dataval_32 >> 16) & 0xff,
                    (dataval_32 >> 8) & 0xff, dataval_32 & 0xff);
            }
            break;

        case VNX_IPMASK:
            if (DevNotLocked(devid))
            {
                sprintf(lms[devid].netmask, "%d.%d.%d.%d", (dataval_32 >> 24) & 0xff, (dataval_32 >> 16) & 0xff,
                    (dataval_32 >> 8) & 0xff, dataval_32 & 0xff);
            }
            break;

        case VNX_IPGW:
            if (DevNotLocked(devid))
            {
                sprintf(lms[devid].gateway, "%d.%d.%d.%d", (dataval_32 >> 24) & 0xff, (dataval_32 >> 16) & 0xff,
                    (dataval_32 >> 8) & 0xff, dataval_32 & 0xff);
            }
            lms[devid].deviceready = 1; // Set Device Ready after init calls
            break;
    
        case VNX_FREQUENCY:
            if (DevNotLocked(devid))
            {
                lms[devid].Frequency = dataval_32;
            }
            break;

        case VNX_FDWELL:
            if (DevNotLocked(devid))
            {
                lms[devid].DwellTime = dataval_32;
            }
            break;

        case VNX_AIDLE:
            if (DevNotLocked(devid))
            {
                lms[devid].SweepIdle = dataval_32;
            }
            break;

        case VNX_FSTART:
            if (DevNotLocked(devid))
            {
                lms[devid].StartFrequency = dataval_32;
            }
            break;

        case VNX_FSTOP:
            if (DevNotLocked(devid))
            {
                lms[devid].EndFrequency = dataval_32;
            }
            break;

        case VNX_FSTEP:
            if (DevNotLocked(devid))
            {
                lms[devid].FrequencyStep = dataval_32;
            }
            break;

        case VNX_SWEEP:
            if (DevNotLocked(devid))
            {
                // -- update the sweep status bits --
                if (dataval_16 & (SWP_ONCE | SWP_CONTINUOUS))		// are we sweeping?
                {
                    lms[devid].DevStatus = lms[devid].DevStatus | SWP_ACTIVE;
                }
                else
                {
                    lms[devid].DevStatus = lms[devid].DevStatus & ~SWP_ACTIVE;
                }

                // -- fill in the SWP_UP status bit
                if (dataval_16 & (SWP_DIRECTION))					// are we sweeping downwards?
                {
                    lms[devid].DevStatus = lms[devid].DevStatus & ~SWP_UP;
                }
                else
                {
                    lms[devid].DevStatus = lms[devid].DevStatus | SWP_UP;
                }

                // -- fill in the continuous sweep bit
                if (dataval_16 & (SWP_CONTINUOUS))					// are we in continuous sweep mode
                {
                    lms[devid].DevStatus = lms[devid].DevStatus | SWP_REPEAT;
                }
                else
                {
                    lms[devid].DevStatus = lms[devid].DevStatus & ~SWP_REPEAT;
                }

                // -- fill in the sweep type bit
                if (dataval_16 & (SWP_BIDIR))						// are we in bi-directional sweep mode
                {
                    lms[devid].DevStatus = lms[devid].DevStatus | SWP_BIDIRECTIONAL;
                }
                else
                {
                    lms[devid].DevStatus = lms[devid].DevStatus & ~SWP_BIDIRECTIONAL;
                }
            }
            break;

        case VNX_OFFTIME:
            if (DevNotLocked(devid))
            {
                lms[devid].PulseOffTime = dataval_32;

                if ((lms[devid].PulseOffTime & 0xF0000000) == 0x10000000)
                {
                    lms[devid].PulseOffTimeSeconds = ((float)((lms[devid].PulseOffTime & 0x0FFFFFFF)) * 2.083333e-8);
                }
                else
                {
                    lms[devid].PulseOffTimeSeconds = ((float)((lms[devid].PulseOffTime & 0x0FFFFFFF)) * 1.00000e-6);
                }
            }
            break;

        case VNX_ONTIME:
            if (DevNotLocked(devid))
            {
                lms[devid].PulseOnTime = dataval_32;

                if ((lms[devid].PulseOnTime & 0xF0000000) == 0x10000000)
                {
                    lms[devid].PulseOnTimeSeconds = ((float)((lms[devid].PulseOnTime & 0x0FFFFFFF)) * 2.083333e-8);
                }
                else
                {
                    lms[devid].PulseOnTimeSeconds = ((float)((lms[devid].PulseOnTime & 0x0FFFFFFF)) * 1.00000e-6);
                }
            }
            break;

        case VNX_PULSE_MODE:
            if (DevNotLocked(devid))
            {
                if (dataval_16 & 0x80)					// the bit is true for disabled fast PWM
                {
                    lms[devid].DevStatus = lms[devid].DevStatus & ~FAST_PULSE_OPTION;
                }
                else											// the option is present, enable its use
                {
                    lms[devid].DevStatus = lms[devid].DevStatus | FAST_PULSE_OPTION;
                }

                lms[devid].Modebits = lms[devid].Modebits & ~PWM_MASK;	// clear the PWM bitfield
                // iResponse <<= 8;										// align the PWM bits with their bitfield
                // in the Modebits

                lms[devid].Modebits = lms[devid].Modebits | dataval_16;	// or in the PWM mode bits
            }
            break;

        case VNX_RFMUTE:
            if (DevNotLocked(devid))
            {
                if (dataval_16)
                    lms[devid].Modebits = lms[devid].Modebits | MODE_RFON;
                else
                    lms[devid].Modebits = lms[devid].Modebits & ~MODE_RFON;
            }
            break;

        case VNX_INTOSC:
            if (DevNotLocked(devid))
            {
                if (dataval_16)
                    lms[devid].Modebits = lms[devid].Modebits | MODE_INTREF;
                else
                    lms[devid].Modebits = lms[devid].Modebits & ~MODE_INTREF;
            }
            break;

        case VNX_EXTSWP:
            if (DevNotLocked(devid))
            {
                if (dataval_16)
                    lms[devid].Modebits = lms[devid].Modebits | MODE_EXTSWTRG;
                else
                    lms[devid].Modebits = lms[devid].Modebits & ~MODE_EXTSWTRG;
            }
            break;

        case VNX_PWR:
            if (DevNotLocked(devid))
            {
                lms[devid].Power = dataval_16;
            }
            break;

        case VNX_MAX_PWR:
            if (DevNotLocked(devid))
            {
                lms[devid].MaxPower = dataval_16;
            }
            break;

        case VNX_MINFREQUENCY:
            // For the BLX-403 we know the correct value is 500 Mhz so we use the value we get from the HW to determine
            // what type of firmware the BLX-403 is running. If we get a response in Mhz units we know it is the old firmware
            // and we will have to scale frequency values accordingly.
            if (DevNotLocked(devid))
            {

                if (lms[devid].DevType == HW_BLX403)
                {
                    // Jitendra's BLX firmware returns MinFrequency in 1 Mhz units for some reason
                    if (dataval_32 == 500)
                    {
                        lms[devid].HasOldBLX_FW = true;
                    }

                    // set the known correct value for later use
                    lms[devid].MinFrequency = 50000000U;            // 500 MHz in 10 Hz units.
                }
                else
                {
                    lms[devid].MinFrequency = dataval_32 * 10000;   // our devices return MinFrequency in 100 KHz units
                    // but our API is all in 10 Hz units
                }
            }
            break;

        case VNX_MAXFREQUENCY:
            // For the BLX-403 we know the correct value is 500 Mhz so we use the value we get from the HW to determine
            // what type of firmware the BLX-403 is running. If we get a response in Mhz units we know it is the old firmware
            // and we will have to scale frequency values accordingly.
            if (DevNotLocked(devid))
            {
                if (lms[devid].DevType == HW_BLX403)
                {
                    // Jitendra's BLX firmware returns MinFrequency in 1 Mhz units for some reason
                    if (dataval_32 == 40000)				// 40GHz in 1 MHz units
                    {
                        lms[devid].HasOldBLX_FW = true;
                    }

                    lms[devid].MaxFrequency = 4000000000U;		// 40GHz in 10Hz units
                }
                else
                {
                    lms[devid].MaxFrequency = dataval_32 * 10000;	// same as MinFrequency
                }
            }
            break;

        case VNX_SETSERNUM:
            if (DevNotLocked(devid))
            {
                lms[devid].SerialNumber = dataval_32;
            }
            break;

        case VNX_PROFILEDWELL:
            if (DevNotLocked(devid))
            {
                lms[devid].SequenceDwell = dataval_32;
            }
            break;

        case VNX_PROFILEIDLE:
            if (DevNotLocked(devid))
            {
                lms[devid].SequenceIdle = dataval_32;
            }
            break;

        case VNX_PROFILECOUNT:
            if (DevNotLocked(devid))
            {
                lms[devid].SequenceLength = dataval_16;
            }
            break;

        case VNX_PROFILESTART:
            if (DevNotLocked(devid))
            {
                lms[devid].SequenceStart = dataval_16;
            }
            break;

        case VNX_SETPROFILE:
        {
            int i = msg[7] & 0x7F;  // msg[7] = byteblock[5], lower 7 bits = sequence index

            if (DEBUG_OUT > 3)
                printf("HW reports Sequence %d Entry Frequency = %u\n", i, dataval_32);

            if (i < SEQ_MAX_ENTRIES && DevNotLocked(devid) && dataresp.length == 6)
            {
                lms[devid].Sequence[i].Frequency10Hz = dataval_32; // msg[3–6] = byteblock[0–3]
                lms[devid].Sequence[i].PowerLevel = msg[6];        // byteblock[4]
                lms[devid].Sequence[i].ControlPower = (msg[7] & 0x80) != 0;
            }
            break;
        }

        default:
            break;
    }

    lms[devid].responseready = 1;  // Device Response Received

    return STATUS_OK;
}

/*******************************************************************************
 * \brief    Reads Data from the socket
 ******************************************************************************/
int readdatafromSock(int sockfd, int devid)
{
    int count;
    char *buf_pr;
    char buf_st[512] = {0};
    int index;

    buf_pr = buf_st;
        
    if ((count = read(sockfd, buf_pr, 512)) >= 0) {

//        printf("READ FROM SOCKET(%d): Count-%d\n", sockfd, count);
        // Check Socket data response > 8 then split into 8 bytes chunks and parse the command data
        if(count > HID_REPORT_LENGTH)
        {
            for(index=0; index < count; index+=8)
            {
                if (DEBUG_OUT > 1) printf("Send Frame Data(%d,%d)\n",index, count);
                parseDataFrame(buf_pr+index, 8, sockfd, devid);
            }
        }
        else
            parseDataFrame(buf_pr, count, sockfd, devid);
    }
    return 0;
}

/*******************************************************************************
 * \brief    Callback for Receiver data handler of thread
 ******************************************************************************/
static void *rxdatahandler_callback(void *threadID) {
  int tid, ret = 0;;
  struct pollfd poll_fd[1];
  tid = (int)(uintptr_t)threadID;

  if (DEBUG_OUT > 1) printf("Thread Callback:%d\n",tid);

  for (;;) {
     //TODO: rx messages from CODU server and wait for terminate command
      poll_fd[0].fd = lms[tid].devicesockfd;
      poll_fd[0].events = POLLIN | POLLRDNORM;
      poll_fd[0].revents = 0;
            
      for(;;)
      {
            ret = poll(poll_fd, 1, -1);
            if (ret > 0)
            {
                if (poll_fd[0].revents & POLLIN || poll_fd[0].revents & POLLRDNORM)
                {
                    poll_fd[0].revents = 0;
                    // Read data
                    readdatafromSock(lms[tid].devicesockfd, tid);
                }
            }    
        }
    }
    return 0;
}


//***************************************************************************
//
// Function call to Get the parameter from the device using message
//  
//*****************************************************************************
bool GetParameter(DEVID deviceID, int GetParam)
{
    BYTE VNX_param[6] = {0, 0, 0, 0, 0, 0};

    if(deviceID >=0)
    {
        // Check Device open or not
        if (!isDeviceSockOpen(deviceID))
            return FALSE;

        if (TestMode) return TRUE; // Test mode, no need to send anything

        if (!SendReport(deviceID, GetParam, VNX_param, 0)) {
          return FALSE;
        }       
    }
    else return FALSE;
    return  TRUE;
}



//***************************************************************************
//
// This function is called to add an LMS device to the set of possible devices
// The IP address must be in the IPv4 format: e.g. 192.168.100.1
//
//***************************************************************************
int fnLMS_AddLMSDevice(char* deviceIP)
{
    int i;

    // test if the new IP address is in a valid format
    if (!is_valid_ip(deviceIP))
        return BAD_PARAMETER;

    // test if this IP address is already in our table of devices
    // if so, we just return success - this allows a calling application to
    // add devices without worrying about if a device was already added
    // we don't put devices in lms[0]

    for (i = 1; i < MAXDEVICES; i++)
    {
        if (strcmp(lms[i].deviceip, deviceIP) == 0)
            return STATUS_OK;
    }

    // look for an empty slot to store the new IP address in
    for (i = 1; i < MAXDEVICES; i++)
    {
        if (lms[i].deviceip[0] == '\0')
        {
            strcpy(lms[i].deviceip, deviceIP);
            lms[i].ModelName[0] = '\0';     // put a null string in each model name field
            lms[i].DevType = 0;
            lms[i].DevStatus = 0;           // clear the device status
            lms[i].SerialNumber = 0;        // clear the serial number
            lms[i].devicesockfd = 0;        // clear the device handles
            lms[i].deviceready = 0;
            lms[i].responseready = 0;       // Device Response ready status 

            lmsdevcount++;
            return STATUS_OK;
        }
    }

    // we did not find an empty slot in our table, not much we can do
    // but report the error
    return DEV_TABLE_FULL;

}

//***************************************************************************
//
// This function is called to remove an LMS device from the set of possible devices
// and make its slot available. The device must not be open
//
//***************************************************************************
int fnLMS_RemoveLMSDevice(char* deviceIP)
{
    int i;

    // test if the new IP address is in a valid format
    if (!is_valid_ip(deviceIP))
        return BAD_PARAMETER;

    // look for this IP address in our table of devices
    // if we find it, and the device is not opened, we will remove it
    for (i = 1; i < MAXDEVICES; i++)
    {
        if (strcmp(lms[i].deviceip, deviceIP) == 0)
        {
            if (lms[i].DevStatus & DEV_OPENED)
                return DEVICE_NOT_READY;
            else
            {
                // we can go ahead and remove the device
                lms[i].deviceip[0] = '\0';
                lms[i].ModelName[0] = '\0';
                lms[i].DevType = 0;
                lms[i].DevStatus = 0;
                lms[i].SerialNumber = 0;
                lms[i].devicesockfd = 0;        // clear the device handles
                lms[i].deviceready = 0;
                lms[i].responseready = 0;       // Device Response ready status

                lmsdevcount--;
                return STATUS_OK;
            }
        }
    }

    // if we did not find the IP address, we got a bad parameter
    return BAD_PARAMETER;
}

//***************************************************************************
// 
// Function to find devices and return number of devices that are connected
//
//***************************************************************************
int fnLMS_GetNumDevices(void)
{
    int NumDevices = 0;
    int i;
    int status;

    // See how many devices we can find, or have found before
    if (TestMode) {
        // construct a fake device
        lms[1].DevType = HW_BLX403;				// BLX-403	added 5-3-23
        lms[1].MinFrequency = 50000000;		    // 500 MHz min (10 Hz units)
        lms[1].MaxFrequency = 4000000000;		// 40 GHz max (10 Hz units)
        lms[1].MaxAttenuation = 180;
        lms[1].MaxPower = HW_MAXP;
        lms[1].MinPower = lms[1].MaxPower - lms[1].MaxAttenuation;
        lms[1].SerialNumber = 104030;
        lms[1].DevStatus = lms[1].DevStatus | DEV_CONNECTED;
        strcpy(lms[1].ModelName, "BLX-403");
        lms[1].Frequency = 520000000;
        lms[1].Power = 20;
        lms[1].StartFrequency = 500000000;
        lms[1].EndFrequency = 600000000;
        lms[1].PulseOnTime = 100;
        lms[1].PulseOnTimeSeconds = 0.1;
        lms[1].PulseOffTime = 500;
        lms[1].PulseOffTimeSeconds = 0.5;
        lms[1].DwellTime = 100;
        lms[1].SweepIdle = 1000;
        lms[1].FrequencyStep = 10000000;
        lms[1].SequenceDwell = 100;
        lms[1].SequenceIdle = 1000;


        // construct a second fake device
        lms[2].DevType = HW_BLX403_20;				// BLX-403-20
        lms[2].MinFrequency = 50000000;		    // 500 MHz min (10 Hz units)
        lms[2].MaxFrequency = 4000000000;		// 40 GHz max (10 Hz units)
        lms[2].MaxAttenuation = 200;
        lms[2].MaxPower = HW_MAXP_20;
        lms[2].MinPower = lms[2].MaxPower - lms[2].MaxAttenuation;
        lms[2].SerialNumber = 204032;
        lms[2].DevStatus = lms[2].DevStatus | DEV_CONNECTED;
        strcpy(lms[2].ModelName, "BLX-403-20");
        lms[2].Frequency = 520000000;
        lms[2].Power = 20;
        lms[2].StartFrequency = 500000000;
        lms[2].EndFrequency = 600000000;
        lms[2].PulseOnTime = 100;
        lms[2].PulseOnTimeSeconds = 0.1;
        lms[2].PulseOffTime = 500;
        lms[2].PulseOffTimeSeconds = 0.5;
        lms[2].DwellTime = 100;
        lms[2].SweepIdle = 1000;
        lms[2].FrequencyStep = 10000000;
        lms[2].SequenceDwell = 100;
        lms[2].SequenceIdle = 1000;
    }
    else
    {
        // go look for some real hardware
        status = FindVNXDevices();
        if (status < 0) return BAD_HID_IO;          // If we fail due to some OS issue we at least try to tell the user about it
    }

    // Total up the number of devices we have
    for (i = 1; i < MAXDEVICES; i++) {

        if (lms[i].DevStatus & DEV_CONNECTED) NumDevices++;
    }

    return NumDevices;
}

//***************************************************************************
// 
// ------------ Get info for the connected devices --------------------
//
// ActiveDevices must be an array large enough to hold NumDevices DEVIDs
// The function returns the number of DEVIDs in the array
// 
//***************************************************************************
int fnLMS_GetDevInfo(DEVID* ActiveDevices)
{
    int i;
    int NumDevices = 0;

    if (ActiveDevices == NULL) return 0;        // bad array pointer, no place to put the DEVIDs

    for (i = 1; i < MAXDEVICES; i++) {        // NB -- we never put an active device in lms[0] - so DEVIDs start
        // at 1

        if (lms[i].DevStatus & DEV_CONNECTED) {

            ActiveDevices[NumDevices] = i;
            NumDevices++;
        }
    }

    return NumDevices;

}

//***************************************************************************
//
// Function call to Initialize the LMS Device and Open the socket connection
// to that device for communication 
//  
//*****************************************************************************
int fnLMS_InitDevice(DEVID deviceID) {
    int sockfd;
    int devid;
    int err;
    unsigned int index;
    unsigned int i;

    // Check whether already the socket open for that device
//  if(isDeviceSockOpen(deviceip))
//      return STATUS_ERROR;

    if (TestMode){    
        lms[deviceID].devicesockfd = 1234;
    }
    else {
        // Check if the device is connected
        if (!(lms[deviceID].DevStatus & DEV_CONNECTED)) {
            return DEVICE_NOT_READY;
        }

        if (!is_valid_ip(lms[deviceID].deviceip))
            return BAD_IP_ADDRESS;

        // Get the ip address
        char deviceip[MAX_NETBUFF];
        strcpy(deviceip, lms[deviceID].deviceip);

        // Go ahead and open a handle to the hardware
        sockfd = vnxDeviceOpen(deviceip);
        if (sockfd == DEVICE_NOT_READY) {
            return DEVICE_NOT_READY;
        }
        lms[deviceID].devicesockfd = sockfd;

 //       printf("Socketfd:%d\n",sockfd);
        // Create a device threads for reading the data from the device using sockets
        // err = CreateThread(NULL, 0, rxdatahandler_callback, (void*)(uintptr_t)deviceID, 0, &threads[deviceID]);
        err = pthread_create(&threads[deviceID], NULL, rxdatahandler_callback, (void*)(uintptr_t)deviceID);

        // Get the current status details of the device parameter
        for(index=0; index < sizeof(STATUS_QUERY_ID); index++)
        {
            //printf("INDEX %d\n", index);

            int cmd = STATUS_QUERY_ID[index];

            // Skip sweep-related commands unless the device supports them
            if ((cmd == VNX_FDWELL || cmd == VNX_AIDLE || cmd == VNX_FSTEP ||
                cmd == VNX_PROFILEDWELL || cmd == VNX_PROFILEIDLE ||
                cmd == VNX_PROFILECOUNT || cmd == VNX_PROFILESTART) &&
                (lms[deviceID].SerialNumber <= BLX_HAS_SWEEP || lms[deviceID].HasOldBLX_FW == true))
            {
                continue;  // skip unsupported command
            }

            if (!GetParameter(deviceID, cmd))
                return BAD_HID_IO;
        }

        // we don't fill the sequence cache at init time to speed up the process
        // but we empty the sequence cache, and each subsequent
        // read from the cache will result in a read from the device if necessary.
        if ((lms[deviceID].SerialNumber <= BLX_HAS_SWEEP || lms[deviceID].HasOldBLX_FW == true))
        {
            for (i = 0; i < lms[deviceID].SequenceMaxLength; i++)
            {
                lms[deviceID].Sequence[i].Frequency10Hz = BLX_SEQ_EMPTY;
            }
        }

        // Wait for device to be ready before returing from init call
        while(fnLMS_CheckDeviceReady(deviceID));
    } // end of real device open process case

    // update the status for the device
    lms[deviceID].DevStatus = lms[deviceID].DevStatus | DEV_OPENED;

    // if we got here everything worked OK
    return STATUS_OK;
}


//***************************************************************************
//
// Function call to close the socket of the open device
//  
//*****************************************************************************
int fnLMS_CloseDevice(DEVID deviceID) {
    int sockfd;

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    // Check Device open
    if(isDeviceSockOpen(deviceID))
    {
        // If Device exists then read the data 
        if(deviceID >= 0)
        {
            if (!TestMode){
                sockfd = lms[deviceID].devicesockfd;
                close(sockfd);
            }

            // Mark it closed in our list of devices
            lms[deviceID].DevStatus = lms[deviceID].DevStatus & ~DEV_OPENED;

            // And clean up for next time
            ClearDevCache(deviceID);  // clear the cached variables to the unknown state
            lms[deviceID].DevType = 0;
            lms[deviceID].DevStatus = 0;
            lms[deviceID].DevStatus |= DEV_CONNECTED; // Maintain connected status
            lms[deviceID].devicesockfd = 0;        // clear the device handles
            lms[deviceID].deviceready = 0;
            lms[deviceID].responseready = 0;       // Device Response ready status
        }
        else
            return INVALID_DEVID;
    }
    else
        return DEVICE_NOT_READY;
    return STATUS_OK;

}


//***************************************************************************
//
// Function call to Check Device is Ready or not
//  
//*****************************************************************************
int fnLMS_CheckDeviceReady(DEVID deviceID) {

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
        return DEVICE_NOT_READY;

    // If Device exists then read the data 
    if((deviceID >= 0) && (lms[deviceID].deviceready !=0))
    {
        return STATUS_OK;
    }
    else
        return DEVICE_NOT_READY;
}

// Get the device's status - this function can be used for an open or closed device.
int fnLMS_GetDeviceStatus(DEVID deviceID)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }
    else
    {
        return lms[deviceID].DevStatus;
    }
}

//***************************************************************************
//
// Function call to Get Model name from the device (ASCII clients)
//  
//*****************************************************************************
int fnLMS_GetModelNameA(DEVID deviceID, char* Model_Name) {
    int NumChars = 0;
    int i;

    if (deviceID >= MAXDEVICES) {

        return 0;
    }

    NumChars = (int)strlen(lms[deviceID].ModelName);
    if (NumChars > 10) NumChars = 10;				// defend against disaster for the caller!
    
    if (Model_Name == NULL) return NumChars;		// NULL result pointer, just return the number of chars in the name

    strcpy(Model_Name, lms[deviceID].ModelName);

    Model_Name[NumChars] = 0;        // Null character to terminate the string
    return NumChars;
}

//***************************************************************************
//
// Function call to Get Model name from the device (Unicode clients)
//  
//*****************************************************************************
int fnLMS_GetModelNameW(DEVID deviceID, wchar_t* Model_Name)
{
    int NumChars = 0;
    size_t Converted = 0;

    if (deviceID >= MAXDEVICES) {
        return 0;
    }

    NumChars = (int)strlen(lms[deviceID].ModelName);

    if (NumChars > 10) NumChars = 10;			// defend against disaster for the caller!

    if (Model_Name == NULL) return NumChars;	// NULL result pointer, just return the number of chars in the name

    mbstowcs(Model_Name, lms[deviceID].ModelName, NumChars);

    return NumChars;
}

//***************************************************************************
//
// Function call to Get Serial# from the device
//  
//*****************************************************************************
int fnLMS_GetSerialNumber(DEVID deviceID)
{
    if (deviceID >= MAXDEVICES) {
        return 0;
    }

    return lms[deviceID].SerialNumber;
}

//***************************************************************************
//
// Function call to Get IP Mode from the device
//  
//*****************************************************************************
LVSTATUS fnLMS_GetIPMode(DEVID deviceID, int* respdata){

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
        return DEVICE_NOT_READY;

    // If Device exists then read the data 
    if(deviceID >= 0)
    {
        GetParameter(deviceID, VNX_IPMODE);
        *respdata = lms[deviceID].ipmode;
    }
    else
        return INVALID_DEVID;
    return STATUS_OK;
}

//***************************************************************************
//
// Function call to Get IP Address from the device
//  
//*****************************************************************************
LVSTATUS fnLMS_GetIPAddress(DEVID deviceID, char* respdata) {
    
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
        return DEVICE_NOT_READY;

    // If Device exists then read the data 
    if(deviceID >= 0)
    {
        GetParameter(deviceID, VNX_IPADDR);
        strcpy(respdata, lms[deviceID].ipaddress);
    }
    else
        return INVALID_DEVID;
    return STATUS_OK;
}

//***************************************************************************
//
// Function call to Get Netmask from the device
//  
//*****************************************************************************
LVSTATUS fnLMS_GetNetmask(DEVID deviceID, char* respdata){
    
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
        return DEVICE_NOT_READY;

    // If Device exists then read the data 
    if(deviceID >= 0)
    {
        GetParameter(deviceID, VNX_IPMASK);
        strcpy(respdata, lms[deviceID].netmask);
    }
    else
        return INVALID_DEVID;
    return STATUS_OK;
}

//***************************************************************************
//
// Function call to Get Gateway from the device
//  
//*****************************************************************************
LVSTATUS fnLMS_GetGateway(DEVID deviceID, char* respdata){
    
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
        return DEVICE_NOT_READY;

    // If Device exists then read the data 
    if(deviceID >= 0)
    {
        GetParameter(deviceID, VNX_IPGW);
        strcpy(respdata, lms[deviceID].gateway);
    }
    else
        return INVALID_DEVID;
    return STATUS_OK;
}





// Functions to set parameters

// Set the frequency of the synthesizer - DSS style, in 10Hz units
// RD 5/3/23 internally we use unsigned ints so the DLL can support the 40 GHz BLX-403
//			 this API function was limited by its int frequency to frequency values less than
//			 21.47483647 GHz

LVSTATUS fnLMS_SetFrequency(DEVID deviceID, unsigned int frequency)
{

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }


    LockDev(deviceID, TRUE);		// prevent updates to the frequency variable reported by the device!

    unsigned int old_frequency = lms[deviceID].Frequency;

    if ((frequency >= lms[deviceID].MinFrequency) && (frequency <= lms[deviceID].MaxFrequency)) {

        lms[deviceID].Frequency = frequency;
        if (TestMode) {
            LockDev(deviceID, FALSE);
            return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
        }
    }
    else
    {
        return BAD_PARAMETER;
    }

    // the frequency value is OK, lets send it to the hardware

    unsigned char* ptr = (unsigned char*)&lms[deviceID].Frequency;

    //	printf("hDevice = %x ptr = %x Frequency = %x\n", lms[deviceID].hDevice, ptr, *ptr);


    if (!SendReport(deviceID, VNX_FREQUENCY | VNX_SET, ptr, 4)) {

        lms[deviceID].Frequency = old_frequency;
        LockDev(deviceID, FALSE);	// We're done using the frequency variables, unlock them..
        return BAD_HID_IO;
    }

    LockDev(deviceID, FALSE);		// We're done, let the status reports update frequency (in case of a sweep)
    return STATUS_OK;
}


// Set the sweep start frequency in 10Hz units
LVSTATUS fnLMS_SetStartFrequency(DEVID deviceID, unsigned int startfrequency)
{
    unsigned int newStartFrequency;

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    newStartFrequency = startfrequency;

    unsigned int old_startfrequency = lms[deviceID].StartFrequency;

    if ((newStartFrequency >= lms[deviceID].MinFrequency) && (newStartFrequency <= lms[deviceID].MaxFrequency)) {

        lms[deviceID].StartFrequency = newStartFrequency;
        if (TestMode) {
            return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
        }
    }
    else
    {
        return BAD_PARAMETER;
    }

    // the sweep start frequency value is OK, lets send it to the hardware

    unsigned char* ptr = (unsigned char*)&lms[deviceID].StartFrequency;

    //	printf("hDevice = %x ptr = %x StartFrequency = %x\n", lms[deviceID].hDevice, ptr, *ptr);


    if (!SendReport(deviceID, VNX_FSTART | VNX_SET, ptr, 4)) {

        lms[deviceID].StartFrequency = old_startfrequency;
        return BAD_HID_IO;
    }

    sleep_ms(5);	// delay so we don't get two commands in one frame
    return STATUS_OK;
}

// Set the sweep end frequency in 10Hz units
LVSTATUS fnLMS_SetEndFrequency(DEVID deviceID, unsigned int endfrequency)
{
    unsigned int newEndFrequency;

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    newEndFrequency = endfrequency;

    unsigned int old_endfrequency = lms[deviceID].EndFrequency;

    if ((newEndFrequency >= lms[deviceID].MinFrequency) && (newEndFrequency <= lms[deviceID].MaxFrequency)) {

        lms[deviceID].EndFrequency = newEndFrequency;
        if (TestMode) {
            return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
        }
    }
    else
    {
        return BAD_PARAMETER;
    }
    // the sweep end frequency value is OK, lets send it to the hardware

    unsigned char* ptr = (unsigned char*)&lms[deviceID].EndFrequency;

    //	printf("hDevice = %x ptr = %x SweepEndFrequency = %x\n", lms[deviceID].hDevice, ptr, *ptr);


    if (!SendReport(deviceID, VNX_FSTOP | VNX_SET, ptr, 4)) {

        lms[deviceID].EndFrequency = old_endfrequency;
        return BAD_HID_IO;
    }

    sleep_ms(5);	// delay so we don't get two commands in one frame
    return STATUS_OK;
}

// Set the sweep step size in 10Hz units
LVSTATUS fnLMS_SetFrequencyStep(DEVID deviceID, unsigned int frequency_step)
{
    unsigned int newFrequencyStep;

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    newFrequencyStep = frequency_step;

    unsigned int old_frequencystep = lms[deviceID].FrequencyStep;

    if (newFrequencyStep <= lms[deviceID].MaxFrequency) {

        lms[deviceID].FrequencyStep = newFrequencyStep;
        if (TestMode) {
            return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
        }
    }
    else
    {
        return BAD_PARAMETER;
    }

    // the sweep frequency step value is OK, lets send it to the hardware
    unsigned char* ptr = (unsigned char*)&lms[deviceID].FrequencyStep;

    //	printf("hDevice = %x ptr = %x FrequencyStep = %x\n", lms[deviceID].hDevice, ptr, *ptr);

    if (!SendReport(deviceID, VNX_FSTEP | VNX_SET, ptr, 4)) {

        lms[deviceID].FrequencyStep = old_frequencystep;
        return BAD_HID_IO;
    }

    sleep_ms(5);	// delay so we don't get two commands in one frame
    return STATUS_OK;
}

// Set the dwell time at each step of the sweep
LVSTATUS fnLMS_SetDwellTime(DEVID deviceID, int dwelltime)
{

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    int old_dwelltime = lms[deviceID].DwellTime;

    if (dwelltime >= BLX_MIN_DWELLTIME) {

        lms[deviceID].DwellTime = dwelltime;
        if (TestMode) {
            return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
        }
    }
    else
    {
        return BAD_PARAMETER;
    }

    // the dwell time value is OK, lets send it to the hardware
    unsigned char* ptr = (unsigned char*)&lms[deviceID].DwellTime;

    //	printf("hDevice = %x ptr = %x SweepTime = %x\n", lms[deviceID].hDevice, ptr, *ptr);


    if (!SendReport(deviceID, VNX_FDWELL | VNX_SET, ptr, 4)) {

        lms[deviceID].DwellTime = old_dwelltime;
        return BAD_HID_IO;
    }

    sleep_ms(5);	// delay so we don't get two commands in one frame
    return STATUS_OK;
}

// Set the idle time at the end of the sweep
LVSTATUS fnLMS_SetIdleTime(DEVID deviceID, int idletime)
{

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    int old_idletime = lms[deviceID].SweepIdle;

    if (idletime >= BLX_MIN_DWELLTIME) {

        lms[deviceID].SweepIdle = idletime;
        if (TestMode) {
            return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
        }
    }
    else
    {
        return BAD_PARAMETER;
    }

    // the idle time value is OK, lets send it to the hardware
    unsigned char* ptr = (unsigned char*)&lms[deviceID].SweepIdle;

    //	printf("hDevice = %x ptr = %x SweepTime = %x\n", lms[deviceID].hDevice, ptr, *ptr);

    if (!SendReport(deviceID, VNX_AIDLE | VNX_SET, ptr, 4)) {
        lms[deviceID].SweepIdle = old_idletime;
        return BAD_HID_IO;
    }

    sleep_ms(5);	// delay so we don't get two commands in one frame
    return STATUS_OK;
}


// ---- BLX Sequence Functions


// Control the operation of sequences - start, stop, pause, resume
LVSTATUS fnLMS_StartSequence(DEVID deviceID, int control)
{
    BYTE VNX_sweep[] = { 0, 0, 0, 0, 0, 0 };

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    // in test mode we update our internal variables, but don't talk to the real HW
    // Our test mode emulation doesn't track and emulate the state of dynamic commands like this one
    if (TestMode) {
        return STATUS_OK;
    }

    switch (control) {

    case STOP_SEQUENCE:
        VNX_sweep[0] = SEQ_STOP;
        lms[deviceID].SequenceMode = SEQ_STOP;
        break;

    case START_SEQUENCE:
        VNX_sweep[0] = SEQ_ONCE;
        lms[deviceID].SequenceMode = SEQ_ONCE;
        break;

    case REPEAT_SEQUENCE:
        VNX_sweep[0] = SEQ_REPEAT;
        lms[deviceID].SequenceMode = SEQ_REPEAT;
        break;

    case PAUSE_SEQUENCE:
        VNX_sweep[0] = SEQ_PAUSE | (lms[deviceID].SequenceMode & 0x03); // we need to include the one time/repeat control bits
        // since the firmware needs them due to a bug
        break;

    case RESUME_SEQUENCE:
        VNX_sweep[0] = SEQ_RESUME;
        break;

    default: return BAD_PARAMETER;
    }

    //if (Trace_Out > 3)	printf(" sending a start sequence command = %x\n", VNX_sweep[0]);

    if (!SendReport(deviceID, VNX_SWEEP | VNX_SET, VNX_sweep, 1))
    {
        return BAD_HID_IO;
    }

    sleep_ms(5);	// delay so we don't get two commands in one frame
    return STATUS_OK;

}

// Set the dwell time for each element of the sequence
LVSTATUS fnLMS_SetSequenceDwellTime(DEVID deviceID, int dwelltime)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    int old_dwelltime = lms[deviceID].SequenceDwell;

    if (dwelltime >= BLX_MIN_DWELLTIME) {

        lms[deviceID].SequenceDwell = dwelltime;
        if (TestMode) {
            return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
        }
    }
    else
    {
        return BAD_PARAMETER;
    }

    // the dwell time value is OK, lets send it to the hardware
    unsigned char* ptr = (unsigned char*)&lms[deviceID].SequenceDwell;

    //	printf("hDevice = %x ptr = %x Sequence DwellTime = %x\n", lms[deviceID].hDevice, ptr, *ptr);

    if (!SendReport(deviceID, VNX_PROFILEDWELL | VNX_SET, ptr, 4)) {
        lms[deviceID].SequenceDwell = old_dwelltime;
        return BAD_HID_IO;
    }

    sleep_ms(5);	// delay so we don't get two commands in one frame
    return STATUS_OK;
}

// Set the idle time at the end of the sequence
LVSTATUS fnLMS_SetSequenceIdleTime(DEVID deviceID, int idletime)
{

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    int old_idletime = lms[deviceID].SequenceIdle;

    // the idle time can be zero, unlike dwell time which has a minimum due to PLL settling time
    if (idletime >= 0) {

        lms[deviceID].SequenceIdle = idletime;
        if (TestMode) {
            return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
        }
    }
    else
    {
        return BAD_PARAMETER;
    }

    // the idle time value is OK, lets send it to the hardware
    unsigned char* ptr = (unsigned char*)&lms[deviceID].SequenceIdle;

    //	printf("hDevice = %x ptr = %x Sequence Idle Time = %x\n", lms[deviceID].hDevice, ptr, *ptr);

    if (!SendReport(deviceID, VNX_PROFILEIDLE | VNX_SET, ptr, 4)) {
        lms[deviceID].SequenceIdle = old_idletime;
        return BAD_HID_IO;
    }

    sleep_ms(5);	// delay so we don't get two commands in one frame
    return STATUS_OK;
}


// Set the length of the sequence
LVSTATUS fnLMS_SetSequenceCount(DEVID deviceID, int count)
{

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    int old_count = lms[deviceID].SequenceLength;

    // the count can be zero, but no more than the maximum sequence length
    if (count >= 0 && count <= (int)lms[deviceID].SequenceMaxLength) {

        lms[deviceID].SequenceLength = count;
        if (TestMode) {
            return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
        }
    }
    else
    {
        return BAD_PARAMETER;
    }

    // the count value is OK, lets send it to the hardware
    unsigned char* ptr = (unsigned char*)&lms[deviceID].SequenceLength;

    //	printf("hDevice = %x ptr = %x Sequence Length = %x\n", lms[deviceID].hDevice, ptr, *ptr);

    if (!SendReport(deviceID, VNX_PROFILECOUNT | VNX_SET, ptr, 4)) {
        lms[deviceID].SequenceLength = old_count;
        return BAD_HID_IO;
    }

    sleep_ms(5);	// delay so we don't get two commands in one frame
    return STATUS_OK;
}

// Set the starting element of the sequence. It is zero based
LVSTATUS fnLMS_SetSequenceStart(DEVID deviceID, int start)
{

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    int old_start = lms[deviceID].SequenceStart;

    // the starting element can run from 0 to Max Length - 1
    if (start >= 0 && start <= (int)lms[deviceID].SequenceMaxLength - 1) {

        lms[deviceID].SequenceStart = start;
        if (TestMode) {
            return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
        }
    }
    else
    {
        return BAD_PARAMETER;
    }

    // the start value is OK, lets send it to the hardware
    unsigned char* ptr = (unsigned char*)&lms[deviceID].SequenceStart;

    //	printf("hDevice = %x ptr = %x Sequence Length = %x\n", lms[deviceID].hDevice, ptr, *ptr);

    if (!SendReport(deviceID, VNX_PROFILESTART | VNX_SET, ptr, 4)) {
        lms[deviceID].SequenceStart = old_start;
        return BAD_HID_IO;
    }

    sleep_ms(5);	// delay so we don't get two commands in one frame
    return STATUS_OK;
}

// set an element in the sequence
// the index runs from 0 to 49 for the BLX devices
// frequency is in 10Hz units
// powerlevel is in 0.25 db units, and is the absolute value for the API call although we send the relative attenuation to the HW
// the pwr_control flag is true when the sequence element controls both frequency and power level.
// if the pwr_control flag is false then only the frequency is controlled.
LVSTATUS fnLMS_SetSequenceElement(DEVID deviceID, int index, unsigned int frequency, int powerlevel, bool pwr_control)
{
    BLX_SEQUENCE tmp_entry;
    BYTE VNX_command[] = { 0, 0, 0, 0, 0, 0 };
    BYTE rel_power;

    // validate our inputs
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    if (index < 0 || index >= (int)lms[deviceID].SequenceMaxLength) {
        return BAD_PARAMETER;
    }

    if (frequency < lms[deviceID].MinFrequency || frequency > lms[deviceID].MaxFrequency) {
        return BAD_PARAMETER;
    }

    if ((powerlevel > lms[deviceID].MaxPower) || (powerlevel < lms[deviceID].MinPower)) {
        return BAD_PARAMETER;
    }


    // Yes, I could have just done an assignment of the array element to the tmp_entry structure
    // but this is easier to validate for me.
    tmp_entry.Frequency10Hz = lms[deviceID].Sequence[index].Frequency10Hz;
    tmp_entry.PowerLevel = lms[deviceID].Sequence[index].PowerLevel;
    tmp_entry.ControlPower = lms[deviceID].Sequence[index].ControlPower;

    // determine the relative attenuation for this powerlevel
    rel_power = (BYTE)(lms[deviceID].MaxPower - powerlevel);


    // update the local cache
    lms[deviceID].Sequence[index].Frequency10Hz = frequency;
    lms[deviceID].Sequence[index].PowerLevel = rel_power;
    lms[deviceID].Sequence[index].ControlPower = pwr_control;

    if (TestMode) {
        return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
    }

    // lets send the new sequence element to the hardware
    VNX_command[0] = (BYTE)(frequency & 0x000000ff);
    VNX_command[1] = (BYTE)((frequency >> 8) & 0x000000ff);
    VNX_command[2] = (BYTE)((frequency >> 16) & 0x000000ff);
    VNX_command[3] = (BYTE)((frequency >> 24) & 0x000000ff);

    VNX_command[4] = rel_power;			// the relative power level (attenuation from max power) in 0.25 db units
    VNX_command[5] = (BYTE)index;		// bits 0 to 6 are the index for the sequence element to update
    if (pwr_control)
    {
        VNX_command[5] |= 0x80;
    }

    //	printf("hDevice = %x Sequence Entry %d Frequency = %x\n", lms[deviceID].hDevice, index, frequency);

    if (!SendReport(deviceID, VNX_SETPROFILE | VNX_SET, VNX_command, 6)) {

        // restore the old entry into the cache
        // using the direct copy here...
        lms[deviceID].Sequence[index] = tmp_entry;
        return BAD_HID_IO;
    }

    sleep_ms(5);	// delay so we don't get two commands in one frame
    return STATUS_OK;
}


// functions to get the components of a sequence element
unsigned int fnLMS_GetSeqElementFrequency(DEVID deviceID, int index)
{
    // validate our inputs
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID_EX;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY_EX;
    }

    // -- make sure the index is valid --
    if (index < 0 || index >(int)lms[deviceID].SequenceMaxLength - 1)
    {
        return BAD_PARAMETER_EX;
    }

    // see if we have the requested sequence element in our cache
    if (lms[deviceID].Sequence[index].Frequency10Hz != 0)
    {
        // we have a cached value
        return lms[deviceID].Sequence[index].Frequency10Hz;
    }
    else
    {
        // we need to read this value from the hardware
        if (GetSequenceElement(deviceID, index))
        {
            return lms[deviceID].Sequence[index].Frequency10Hz;
        }
        else
        {
            // for some reason the read from the HW failed
            return BAD_HID_IO_EX;
        }
    }
}


int fnLMS_GetSeqElementPower(DEVID deviceID, int index)
{
    int result;
    int abs_power;

    result = WF_GetSequenceElement(deviceID, index);

    if (result == STATUS_OK)
    {
        abs_power = lms[deviceID].MaxPower - lms[deviceID].Sequence[index].PowerLevel;

        return abs_power;
    }
    else
    {
        return result;
    }
}
int fnLMS_GetSeqElementPwrControl(DEVID deviceID, int index)
{
    int result;

    result = WF_GetSequenceElement(deviceID, index);

    if (result == STATUS_OK)
    {
        if (lms[deviceID].Sequence[index].ControlPower)
        {
            return 1;
        }
        else
        {
            return 0;
        }
    }
    else
    {
        return result;
    }
}


// this worker function is used by each of the functions which return a component of the sequence element
// it checks to see if the selected element exists in the cache, and reads it from the HW if necessary
// it returns 0 for SUCCESS == STATUS_OK, otherwise an error code if it failed
//
int WF_GetSequenceElement(DEVID deviceID, int index)
{
    // validate our inputs
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    // -- make sure the index is valid --
    if (index < 0 || index >(int)lms[deviceID].SequenceMaxLength - 1)
    {
        return BAD_PARAMETER;
    }

    // see if we have the requested sequence element in our cache
    if (lms[deviceID].Sequence[index].Frequency10Hz != 0)
    {
        // we have a cached value
        return STATUS_OK;
    }
    else
    {
        // we need to read this sequence entry from the hardware
        if (GetSequenceElement(deviceID, index))
        {
            return STATUS_OK;
        }
        else
        {
            // for some reason the read from the HW failed
            return BAD_HID_IO;
        }
    }
}

bool GetSequenceElement(DEVID deviceID, int index)
{
    int max_index;
    clock_t start_time;
    int timedout = 0;
    const int timeout_ms = 1000;

    if (deviceID < 0 || deviceID >= MAXDEVICES)
        return FALSE;

    max_index = lms[deviceID].SequenceMaxLength - 1;	// get the size of the sequence buffer for this device 

    if (index < 0 || index > max_index) return FALSE;	// bail out on a bad index...

    BYTE VNX_command[6] = { 0, 0, 0, 0, 0, 0 };			// the byteblock for our command 
    VNX_command[5] = (BYTE)index;

    lms[deviceID].responseready = 0;

    // --- send the command out to the device --
    if (!SendReport(deviceID, VNX_SETPROFILE | VNX_GET, VNX_command, 6))
        return FALSE;

    // --- then we wait for the read thread's parser to signal that it got the response
    start_time = clock();

    while (!lms[deviceID].responseready && !timedout) {
        sleep_ms(10); // yield CPU for 10 ms

        clock_t elapsed_ticks = clock() - start_time;
        int elapsed_ms = (int)((elapsed_ticks * 1000) / CLOCKS_PER_SEC);
        if (elapsed_ms > timeout_ms)
            timedout = 1;
    }

    return (timedout == 0);
}

int fnLMS_GetSequenceStart(DEVID deviceID)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }
    
    return lms[deviceID].SequenceStart;
}

int fnLMS_GetSequenceCount(DEVID deviceID)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    return lms[deviceID].SequenceLength;
}

int fnLMS_GetSequenceDwellTime(DEVID deviceID)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    return lms[deviceID].SequenceDwell;
}

int fnLMS_GetSequenceIdleTime(DEVID deviceID)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    return lms[deviceID].SequenceIdle;
}

// Set the output power level
LVSTATUS fnLMS_SetPowerLevel(DEVID deviceID, int powerlevel)
{
    BYTE VNX_command[] = { 0, 0, 0, 0, 0, 0 };

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    int old_powerlevel = lms[deviceID].Power;


    // We use an absolute power level for the API, so the attenuation is computed from the value set by the user

    // First off, check if the desired power setting is in range

    //if (IO_Trace > 3)
       //printf("In SetPowerLevel, MaxPower = %d, MinPower = %d\n", lms[deviceID].MaxPower, lms[deviceID].MinPower);

    if ((powerlevel > lms[deviceID].MaxPower) || (powerlevel < lms[deviceID].MinPower))
    {
        return BAD_PARAMETER;		// power setting is out of range, bail out
    }

    // We set the attenuation relative to MaxPower, so we have to subtract the power level value to get the attenuation
    // for powerlevel settings below 0 db we end up adding the value since it is negative.

    powerlevel = (lms[deviceID].MaxPower - powerlevel);

    //if (IO_Trace > 3) printf("In SetPowerLevel, powerlevel = %d \n", powerlevel);

    // At this point our powerlevel value has to be valid, since we tested the absolute setting against our range
    // We could test if powerlevel < lms[deviceID].MaxAttenuation, but we already know that is true
    // So as of 2/20/20 we no longer make that test

    lms[deviceID].Power = powerlevel;
    if (TestMode) {
        return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
    }

    VNX_command[0] = (BYTE)(lms[deviceID].Power & 0xFF);		// clip for safety and turn into a byte in our command buffer

    // SendReport byte count changed from 4 to 1, which I think is what we want
    //if (Trace_Out > 3)	printf("DeviceType = %d HW PowerLevel = %x\n", lms[deviceID].DevType, VNX_command[0]);

    if (!SendReport(deviceID, VNX_PWR | VNX_SET, VNX_command, 1)) {

        lms[deviceID].Power = old_powerlevel;
        return BAD_HID_IO;
    }

    return STATUS_OK;
}

// Set the RF output on or off
LVSTATUS fnLMS_SetRFOn(DEVID deviceID, bool on)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    BYTE VNX_command[] = { 0, 0, 0, 0, 0, 0 };	// RD 2/23/17 changed to 6 bytes, the actual command buffer size

    if (on) {

        lms[deviceID].Modebits = lms[deviceID].Modebits & ~MODE_RFON;
        lms[deviceID].Modebits = lms[deviceID].Modebits | MODE_RFON;
        VNX_command[0] = 1;
    }
    else
    {
        lms[deviceID].Modebits = lms[deviceID].Modebits & ~MODE_RFON;
        VNX_command[0] = 0;
    }

    if (TestMode) {
        return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
    }

    if (!SendReport(deviceID, VNX_RFMUTE | VNX_SET, VNX_command, 1)) {

        return BAD_HID_IO;
    }

    return STATUS_OK;
}

// Set the state of the frequency reference - internal or external
// if internal = TRUE then the internal oscillator is used.
LVSTATUS fnLMS_SetUseInternalRef(DEVID deviceID, bool internal)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    BYTE VNX_command[] = { 0, 0, 0, 0, 0, 0 };

    if (internal) {

        lms[deviceID].Modebits = lms[deviceID].Modebits | MODE_INTREF;
        VNX_command[0] = 1;
    }
    else
    {
        lms[deviceID].Modebits = lms[deviceID].Modebits & ~MODE_INTREF;
        VNX_command[0] = 0;
    }


    if (TestMode) {
        return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
    }

    if (!SendReport(deviceID, VNX_INTOSC | VNX_SET, VNX_command, 1)) {

        return BAD_HID_IO;
    }

    return STATUS_OK;
}



// Set the sweep direction -- "up" is TRUE to sweep upwards
LVSTATUS fnLMS_SetSweepDirection(DEVID deviceID, bool up)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }
    if (up)
    {
        lms[deviceID].Modebits = lms[deviceID].Modebits & ~SWP_DIRECTION;	// sweep direction up (bit == 0)
    }
    else
    {
        lms[deviceID].Modebits = lms[deviceID].Modebits | SWP_DIRECTION;	// sweep direction downwards
    }


    return STATUS_OK;
}


// Set the sweep mode -- mode = TRUE for repeated sweep, FALSE for one time sweep
LVSTATUS fnLMS_SetSweepMode(DEVID deviceID, bool mode)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    if (mode)
    {
        lms[deviceID].Modebits = lms[deviceID].Modebits | SWP_CONTINUOUS;	// Repeated sweep
        lms[deviceID].Modebits = lms[deviceID].Modebits & ~SWP_ONCE;
    }
    else
    {
        lms[deviceID].Modebits = lms[deviceID].Modebits | SWP_ONCE;			// one time sweep
        lms[deviceID].Modebits = lms[deviceID].Modebits & ~SWP_CONTINUOUS;
    }


    return STATUS_OK;
}


// Set the sweep type -- swptype =	TRUE for bi-directional ramp up and ramp down sweep,
//									FALSE for ramp type uni-directional sweep
LVSTATUS fnLMS_SetSweepType(DEVID deviceID, bool swptype)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    if (swptype)
    {
        lms[deviceID].Modebits = lms[deviceID].Modebits | SWP_BIDIR;	// sweep bidirectionally 
    }
    else
    {
        lms[deviceID].Modebits = lms[deviceID].Modebits & ~SWP_BIDIR;	// sawtooth shaped sweep envelope
    }


    return STATUS_OK;
}


// Set the state of the sweep trigger control - internal or external
// if external = TRUE then the external sweep trigger input is enabled
LVSTATUS fnLMS_SetUseExternalSweepTrigger(DEVID deviceID, bool external)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    BYTE VNX_command[] = { 0, 0, 0, 0, 0, 0 };

    if (external) {

        lms[deviceID].Modebits = lms[deviceID].Modebits | MODE_EXTSWTRG;
    }
    else
    {
        lms[deviceID].Modebits = lms[deviceID].Modebits & ~MODE_EXTSWTRG;
    }

    VNX_command[0] = (lms[deviceID].Modebits & MODE_EXTSWTRG) >> 10;		// move our mode bit over, 0x01 is use external

    if (TestMode) {
        return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
    }

    if (!SendReport(deviceID, VNX_EXTSWP | VNX_SET, VNX_command, 1)) {

        return BAD_HID_IO;
    }

    sleep_ms(5);	// delay so we don't get two commands in one frame
    return STATUS_OK;
}


// Start the sweep
//
LVSTATUS fnLMS_StartSweep(DEVID deviceID, bool go)
{
    int tmp_mask;

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    BYTE VNX_sweep[] = { 0, 0, 0, 0, 0, 0 };

    if (go)
    {
        tmp_mask = MODE_SWEEP;

        VNX_sweep[0] = (BYTE)lms[deviceID].Modebits & tmp_mask;
    }
    else
    {
        VNX_sweep[0] = 0;
    }

    if (TestMode) {
        return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
    }

    //if (Trace_Out > 3)	printf(" sending a sweep command = %x\n", VNX_sweep[0]);

    if (!SendReport(deviceID, VNX_SWEEP | VNX_SET, VNX_sweep, 1))
    {
        return BAD_HID_IO;
    }

    sleep_ms(5);	// delay so we don't get two commands in one frame
    return STATUS_OK;
}


// Set the state of the pulsed mode output control - internal or external
// if external = TRUE then the external pulse modulation input is enabled
LVSTATUS fnLMS_SetUseExternalPulseMod(DEVID deviceID, bool external)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    BYTE VNX_command[] = { 0, 0, 0, 0, 0, 0 };

    if (external) {

        lms[deviceID].Modebits = lms[deviceID].Modebits | MODE_EXTPWM;
    }
    else
    {
        lms[deviceID].Modebits = lms[deviceID].Modebits & ~MODE_EXTPWM;
    }

    VNX_command[0] = (lms[deviceID].Modebits & PWM_MASK) >> 8;

    if (TestMode) {
        return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
    }

    if (!SendReport(deviceID, VNX_PULSE_MODE | VNX_SET, VNX_command, 1)) {

        return BAD_HID_IO;
    }

    sleep_ms(5);	// delay so we don't get two commands in one frame
    return STATUS_OK;
}

// Turn on or off the internal pulsed mode output control.
// if on = TRUE then the internal pulse modulation input is enabled
LVSTATUS fnLMS_EnableInternalPulseMod(DEVID deviceID, bool on)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    BYTE VNX_command[] = { 0, 0, 0, 0, 0, 0 };

    if (on) {

        lms[deviceID].Modebits = lms[deviceID].Modebits | MODE_PWMON;
    }
    else
    {
        lms[deviceID].Modebits = lms[deviceID].Modebits & ~MODE_PWMON;
    }

    VNX_command[0] = (lms[deviceID].Modebits & PWM_MASK) >> 8;

    if (TestMode) {
        return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
    }

    if (!SendReport(deviceID, VNX_PULSE_MODE | VNX_SET, VNX_command, 1)) {

        return BAD_HID_IO;
    }

    sleep_ms(5);	// delay so we don't get two commands in one frame
    return STATUS_OK;
}



// This function takes two float arguments - pulse on time in seconds and pulse rep time in seconds
// The BLX has a different higher resolution HW Pulse Modulation clock so we use it when possible
// even though the BLX firmware can accept the other pulse timing parameters.
//
// RD 10/21/23	I have merged in the 7/7/23 fixes from the LMS DLL which fix a bug and improve the accuracy
//				of the calculations

LVSTATUS fnLMS_SetFastPulsedOutput(DEVID deviceID, float pulseontime, float pulsereptime, bool on)
{
    int temp_pulseontime;
    int temp_pulseofftime;
    double reptimetest;
    double HP_pulseontime;					// trying to promote the args to improve accuracy
    double HP_pulsereptime;

    BYTE VNX_command[] = { 0, 0, 0, 0, 0, 0 };

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    if (!(lms[deviceID].DevStatus & FAST_PULSE_OPTION))
    {
        return BAD_PARAMETER;
    }

    float old_pulseontime = lms[deviceID].PulseOnTimeSeconds;
    float old_pulseofftime = lms[deviceID].PulseOffTimeSeconds;

    if (pulsereptime <= pulseontime)		// the on time has to be less than the repetition time
    {
        return BAD_PARAMETER;
    }

    // ------ first we have to convert from the floating point times to our integers ---------------------
    // 
    // we have to pick the range of units depending on whether or not we can use the 48MHz, 1 MHz or 1Ms clocks
    // for the BLX we can also choose a 120 MHz clock

    //	printf ("pulseontime = %f, pulsereptime = %f\n", pulseontime, pulsereptime);

    HP_pulseontime = (double)pulseontime;
    HP_pulsereptime = (double)pulsereptime;

    if (lms[deviceID].DevType == HW_BLX403 ||
        lms[deviceID].DevType == HW_BLX403_20 ||
        lms[deviceID].DevType == HW_BLX223)
    {
        // for a BLX device we have a 24 bit 120 MHz counter. We use it for all pulse modulation with a repetition time
        // less than 0.10 seconds.
        reptimetest = 1.0000000E-1;

        if (HP_pulsereptime <= reptimetest)	// use the 120 MHz clock
        {
            if ((HP_pulseontime + 100.0e-9 > HP_pulsereptime))	// the on time plus our minimum pulse width has to be less than the repetition time
            {
                return BAD_PARAMETER;
            }

            // generate the integer on time
            //		printf ("using 120MHz clock\n");

            temp_pulseontime = (int)round(120 * (HP_pulseontime * 1.0e+6));
            //		printf ("temp_pulseontime = %d", temp_pulseontime);

            temp_pulseofftime = (int)round(120 * (HP_pulsereptime * 1.0e+6)) - temp_pulseontime;
            //		printf ("temp_pulseofftime = %d", temp_pulseofftime);

            temp_pulseontime |= PM120MHz;

            if (temp_pulseofftime <= 0) return BAD_PARAMETER;
        }
        else		// for rep time > 100 ms we use the software timer
        {

            if ((HP_pulseontime + .001 >= HP_pulsereptime))		// the on time plus one clock has to be less than the repetition time
            {
                return BAD_PARAMETER;
            }

            if (HP_pulsereptime > 1000) return BAD_PARAMETER;	// maximum time is 1000 seconds

            //		printf ("using 1 ms. timer\n");

            // ---- we represent the time in 1 microsecond units --------
            temp_pulseontime = (int)round(HP_pulseontime * 1.0e+6);
            temp_pulseofftime = (int)round(HP_pulsereptime * 1.0e+6) - temp_pulseontime;

            if (temp_pulseofftime <= 0) return BAD_PARAMETER;

        }
    }
    else
    {
        // we have a typical LMS device, so use one of its pulse modulation operating modes
        reptimetest = 1.041688E-3;

        if (HP_pulsereptime < reptimetest)						// use the 48MHz clock
        {
            if ((HP_pulseontime + 5.0e-9 > HP_pulsereptime))	// the on time plus one clock has to be less than the repetition time
            {
                return BAD_PARAMETER;
            }

            // printf ("using 48MHz clock\n");

            // generate the integer on time
            temp_pulseontime = (int)round(48 * (HP_pulseontime * 1.0e+6));
            //		printf ("temp_pulseontime = %d", temp_pulseontime);

            // generate the integer off time
            temp_pulseofftime = (int)round(48 * (HP_pulsereptime * 1.0e+6)) - temp_pulseontime;
            //		printf ("temp_pulseofftime = %d", temp_pulseofftime);

            // mark the value with the encoding in the top 4 bits
            temp_pulseontime |= PM48Mhz;

            if (temp_pulseofftime <= 0) return BAD_PARAMETER;
        }
        else if (HP_pulsereptime > .001 && HP_pulsereptime <= .050)	// use the 1MHz clock
        {
            if ((HP_pulseontime + 1.0e-6 > HP_pulsereptime))		// the on time plus one clock has to be less than the repetition time
            {
                return BAD_PARAMETER;
            }

            //		printf ("using 1MHz clock\n");

            temp_pulseontime = (int)round(HP_pulseontime * 1.0e+6);
            temp_pulseofftime = (int)round(HP_pulsereptime * 1.0e+6) - temp_pulseontime;

            if (temp_pulseofftime <= 0) return BAD_PARAMETER;
        }
        else	// for rep time > 50 ms we use the software timer (its really the same as the above case in terms of units)
        {

            if ((HP_pulseontime + .001 > HP_pulsereptime))	// the on time plus one clock has to be less than the repetition time
            {
                return BAD_PARAMETER;
            }

            if (HP_pulsereptime > 1000) return BAD_PARAMETER;	// maximum time is 1000 seconds

            //		printf ("using 1 ms. timer\n");

            // ---- we represent the time in 1 microsecond units --------
            temp_pulseontime = (int)round(HP_pulseontime * 1.0e+6);
            temp_pulseofftime = (int)round(HP_pulsereptime * 1.0e+6) - temp_pulseontime;

            if (temp_pulseofftime <= 0) return BAD_PARAMETER;
        }

    }

    //	printf ("pulseontime = %d, pulseofftime = %d\n", (temp_pulseontime & 0xFFFFFF), temp_pulseofftime);

    // At this point we can update our local copies of the original floating point on and off time in seconds
    // We'll restore the old values if somehow the hardware I/O fails

    lms[deviceID].PulseOnTimeSeconds = pulseontime;
    lms[deviceID].PulseOffTimeSeconds = pulsereptime - pulseontime;

    //if (Trace_Out > 3)	printf("PulseOnTimeSeconds = %f, PulseOffTimeSeconds = %f\n", lms[deviceID].PulseOnTimeSeconds, lms[deviceID].PulseOffTimeSeconds);

    // Now send the parameters to the device if we aren't in test mode

    if (TestMode)
    {
        if (on) {				// keep the mode bits in sync

            lms[deviceID].Modebits = lms[deviceID].Modebits | MODE_PWMON;
        }
        else
        {
            lms[deviceID].Modebits = lms[deviceID].Modebits & ~MODE_PWMON;
        }

        return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
    }

    // First we disable any active fast pulse mode operation, including external mode ...
    VNX_command[0] = 0;			// pulse mode off, internal pulse modulation selected

    if (!SendReport(deviceID, VNX_PULSE_MODE | VNX_SET, VNX_command, 1))
    {
        // -- our IO failed, so leave the old settings and bail out --
        //	  we can't do much about restoring the pulse mode state...
        lms[deviceID].PulseOnTimeSeconds = old_pulseontime;
        lms[deviceID].PulseOffTimeSeconds = old_pulseofftime;

        return BAD_HID_IO;
    }

    sleep_ms(5);	// delay so we don't get two commands in one frame

    // Then we send the on time, followed by the off time
    unsigned char* ptr = (unsigned char*)&temp_pulseontime;

    //if (Trace_Out > 3)	printf("hDevice = 0x%x ptr = 0x%x Pulse On Time LSByte = 0x%x PulseOnTime = %d\n", lms[deviceID].hDevice, ptr, *ptr, temp_pulseontime);

    if (!SendReport(deviceID, VNX_ONTIME | VNX_SET, ptr, 4)) {

        // -- our IO failed, so leave the old settings and bail out --
        lms[deviceID].PulseOnTimeSeconds = old_pulseontime;
        lms[deviceID].PulseOffTimeSeconds = old_pulseofftime;
        return BAD_HID_IO;
    }

    sleep_ms(5);	// delay so we don't get two commands in one frame

    ptr = (unsigned char*)&temp_pulseofftime;

    //if (Trace_Out > 3)	printf("hDevice = 0x%x ptr = 0x%x Pulse Off Time LSByte = 0x%x PulseOffTime = %d\n", lms[deviceID].hDevice, ptr, *ptr, temp_pulseofftime);

    if (!SendReport(deviceID, VNX_OFFTIME | VNX_SET, ptr, 4)) {

        // -- we're in a pickle here, we set the new pulse on time, but failed on the new off time setting
        //    so our state variables may not be viable value wise, but since talking to the device is failing
        //    we really can't do much about it!

        lms[deviceID].PulseOffTimeSeconds = old_pulseofftime;
        return BAD_HID_IO;
    }

    sleep_ms(5);	// delay so we don't get two commands in one frame

    // -- time to activate the pulse mode operation with the new settings --
    if (on) {

        lms[deviceID].Modebits = lms[deviceID].Modebits | MODE_PWMON;
    }
    else
    {
        lms[deviceID].Modebits = lms[deviceID].Modebits & ~MODE_PWMON;
    }

    VNX_command[0] = (lms[deviceID].Modebits & PWM_MASK) >> 8;

    if (!SendReport(deviceID, VNX_PULSE_MODE | VNX_SET, VNX_command, 1)) {

        // -- a failure here leaves the settings intact, and in sync, except for the mode bits
        //    probably not worth worrying about trying to restore them since who knows what is
        //    going on below us to cause the failure...
        return BAD_HID_IO;
    }

    //if (Trace_Out > 3)	printf("Set Pulse Mode to: 0x%x\n", VNX_command[0]);

    return STATUS_OK;		// it seems everything turned out OK!
}


// Set the internal pulse modulation pulse on time
// This code converts from the floating point on time value (in seconds) to the ranged integer units
// it does not check for required relationship with off time.

LVSTATUS fnLMS_SetPulseOnTime(DEVID deviceID, float pulseontime)
{
    int temp_pulseontime;

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    if (!(lms[deviceID].DevStatus & FAST_PULSE_OPTION))
    {
        return BAD_PARAMETER;
    }

    float old_pulseontime = lms[deviceID].PulseOnTimeSeconds;

    // check if the pulse on time value is in range, and handle the test mode case to get it out of the way
    if (pulseontime >= 0.10e-6 && pulseontime < 1000) {

        lms[deviceID].PulseOnTimeSeconds = pulseontime;	// optimistically update 
        if (TestMode) {
            return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
        }
    }
    else
    {
        return BAD_PARAMETER;		// we exit here if the on time is less than 100ns, of greater than 1000 secs.
    }

    // the BLX-403 has a different HW timer resolultion and range so we handle it separately
    if (lms[deviceID].DevType == HW_BLX403)
    {
        if (pulseontime <= .1)		// use the 120MHz clock
        {
            // generate the integer on time
            temp_pulseontime = (int)round(120 * (pulseontime * 1.0e+6));
            temp_pulseontime |= PM120MHz;
        }
        else
        {
            temp_pulseontime = (int)round(pulseontime * 1.0e+6);
        }
    }
    else
    {
        // the typical LMS devices use either the 48Mhz or 1Mhz encoding
        if (pulseontime <= .001)	// use the 48MHz clock
        {
            // generate the integer on time
            temp_pulseontime = (int)round(48 * (pulseontime * 1.0e+6));
            temp_pulseontime |= PM48Mhz;
        }
        else
        {
            temp_pulseontime = (int)round(pulseontime * 1.0e+6);
        }

    }

    // the pulse on time value is OK, lets send it to the hardware
    unsigned char* ptr = (unsigned char*)&temp_pulseontime;

    //if (Trace_Out > 3)	printf("hDevice = %x ptr = %x Pulse On Time = %x\n", lms[deviceID].hDevice, ptr, *ptr);

    if (!SendReport(deviceID, VNX_ONTIME | VNX_SET, ptr, 4)) {

        lms[deviceID].PulseOnTimeSeconds = old_pulseontime; // IO failed, restore the old value	
        return BAD_HID_IO;
    }

    // we keep a copy of the scaled integer representation around too for internal use
    lms[deviceID].PulseOnTime = temp_pulseontime;

    return STATUS_OK;
}


// Set the internal pulse modulation pulse off time
// Same as SetPulseOnTime as far as its parameters are concerned
// These direct set functions must be used carefully since the calling program is responsible
// for ensuring that they are called in the correct order, and that the units match

LVSTATUS fnLMS_SetPulseOffTime(DEVID deviceID, float pulseofftime)
{
    int temp_pulseofftime;

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    if (!(lms[deviceID].DevStatus & FAST_PULSE_OPTION))
    {
        return BAD_PARAMETER;
    }

    float old_pulseofftime = lms[deviceID].PulseOffTimeSeconds;

    // ------ make sure parameter is in range ----------
    if (pulseofftime >= 150.0e-9 && pulseofftime < 1000) {

        lms[deviceID].PulseOffTimeSeconds = pulseofftime;
        if (TestMode) {
            return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
        }
    }
    else
    {
        return BAD_PARAMETER;		// we end up here if the off time is less than 150ns or greater than 1000 sec
    }

    // the BLX-403 has a different HW timer resolultion and range so we handle it separately
    if (lms[deviceID].DevType == HW_BLX403)
    {
        if (pulseofftime <= .1)		// use the 120MHz clock
        {
            // generate the integer off time
            temp_pulseofftime = (int)round(120 * (pulseofftime * 1.0e+6));
        }
        else
        {
            temp_pulseofftime = (int)round(pulseofftime * 1.0e+6);
        }
    }
    else
    {
        if (pulseofftime <= .001)		// use the 48MHz clock
        {
            // generate the integer off time, it does not have the range flags!!
            temp_pulseofftime = (int)round(48 * (pulseofftime * 1.0e+6));
        }
        else
        {
            temp_pulseofftime = (int)round(pulseofftime * 1.0e+6);
        }
    }

    // the pulse on time value is OK, lets send it to the hardware
    unsigned char* ptr = (unsigned char*)&temp_pulseofftime;

    //if (Trace_Out > 3)	printf("hDevice = %x ptr = %x Pulse Off Time = %x\n", lms[deviceID].hDevice, ptr, *ptr);

    if (!SendReport(deviceID, VNX_OFFTIME | VNX_SET, ptr, 4)) {

        lms[deviceID].PulseOffTimeSeconds = old_pulseofftime;
        return BAD_HID_IO;
    }

    // we keep a copy of the scaled integer representation around too for internal use
    lms[deviceID].PulseOffTime = temp_pulseofftime;

    return STATUS_OK;
}


// Save the user settings to flash for autonomous operation
LVSTATUS fnLMS_SaveSettings(DEVID deviceID)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    if (TestMode) {
        return STATUS_OK;		// in test mode we update our internal variables, but don't talk to the real HW
    }

    BYTE VNX_savesettings[] = { 0x42, 0x55, 0x31 }; //three byte key to unlock the user protection.

    if (!SendReport(deviceID, VNX_SAVEPAR | VNX_SET, VNX_savesettings, 3)) {

        return BAD_HID_IO;
    }

    return STATUS_OK;
}

// ------------- Functions to get parameters --------------------- 

// Get the frequency - 0 to 40 GHz
unsigned int fnLMS_GetFrequency(DEVID deviceID)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID_EX;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY_EX;
    }

    // Update our stored frequency value
    if (!TestMode) {
        GetParameter(deviceID, VNX_FREQUENCY);
    }

    return lms[deviceID].Frequency;
}

// Get the sweep start frequency 0 to 40 GHz
unsigned int fnLMS_GetStartFrequency(DEVID deviceID)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID_EX;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY_EX;
    }

    return lms[deviceID].StartFrequency;
}

// Get the sweep end frequency 0 to 40 GHz
unsigned int fnLMS_GetEndFrequency(DEVID deviceID)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID_EX;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY_EX;
    }

    return lms[deviceID].EndFrequency;
}

// Get the frequency step size (in 10Hz units) for the BLX-403
// stepped sweep. The range is 0 to 40 GHz
unsigned int fnLMS_GetFrequencyStep(DEVID deviceID)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID_EX;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY_EX;
    }

    // -- this is a BLX only function --
    if (lms[deviceID].DevType != HW_BLX403 &&
        lms[deviceID].DevType != HW_BLX403_20 &&
        lms[deviceID].DevType != HW_BLX223) {
        return BAD_PARAMETER_EX;
    }

    return lms[deviceID].FrequencyStep;
}


// Get the dwell time per step for BLX stepped sweeps
// the time is in 1ms units
int fnLMS_GetDwellTime(DEVID deviceID)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    // -- this is a BLX only function --
    if (lms[deviceID].DevType != HW_BLX403 &&
        lms[deviceID].DevType != HW_BLX403_20 &&
        lms[deviceID].DevType != HW_BLX223) {
        return BAD_PARAMETER;
    }

    return lms[deviceID].DwellTime;
}

// Get the idle time for repeated BLX-403 stepped sweeps
// the time is in 1ms units
int fnLMS_GetIdleTime(DEVID deviceID)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    // -- this is a BLX only function --
    if (lms[deviceID].DevType != HW_BLX403 &&
        lms[deviceID].DevType != HW_BLX403_20 &&
        lms[deviceID].DevType != HW_BLX223) {
        return BAD_PARAMETER;
    }

    return lms[deviceID].SweepIdle;
}

// Get the pulse on time
float fnLMS_GetPulseOnTime(DEVID deviceID)
{

    if (deviceID >= MAXDEVICES) {
        return F_INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return F_DEVICE_NOT_READY;
    }

    return lms[deviceID].PulseOnTimeSeconds;
}

// Get the pulse off time
float fnLMS_GetPulseOffTime(DEVID deviceID)
{

    if (deviceID >= MAXDEVICES) {
        return F_INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return F_DEVICE_NOT_READY;
    }

    return lms[deviceID].PulseOffTimeSeconds;
}


// Get the state of the RF output - on or off
int fnLMS_GetRF_On(DEVID deviceID)
{

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    if (lms[deviceID].Modebits & MODE_RFON) {

        return 1;
    }
    else
    {
        return 0;
    }
}

// Get the state of the frequency reference - internal or external
int fnLMS_GetUseInternalRef(DEVID deviceID)
{

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    if (lms[deviceID].Modebits & MODE_INTREF) {

        return 1;
    }
    else
    {
        return 0;
    }
}


// Find out if the fast pulse mode option is installed
int fnLMS_GetHasFastPulseMode(DEVID deviceID)
{

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    if (lms[deviceID].DevStatus & FAST_PULSE_OPTION) {

        return 1;
    }
    else
    {
        return 0;
    }
}


// Find out if pulse mode output is active
int fnLMS_GetPulseMode(DEVID deviceID)
{

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    if (lms[deviceID].Modebits & MODE_PWMON) {

        return 1;
    }
    else
    {
        return 0;
    }
}

// Find out if we are using the internal pulse modulation source
int fnLMS_GetUseInternalPulseMod(DEVID deviceID)
{

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    if (lms[deviceID].Modebits & MODE_EXTPWM) {

        return 0;			// Note -- the flag is true for external PWM src, in the DLL API we 
    }						//         report use of the internal source as true.
    else
    {
        return 1;
    }
}


// Find out if we are using the internal sweep trigger source
int fnLMS_GetUseInternalSweepTrigger(DEVID deviceID)
{

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID))
    {
        return DEVICE_NOT_READY;
    }

    if (lms[deviceID].Modebits & MODE_EXTSWTRG) {

        return 0;										// Note -- the flag is true for external trigger src, in the DLL API we 
    }													//         report use of the internal source as true.
    else
    {
        return 1;										// We are using the internal trigger source, so the flag is not set
    }
}

// Get the maximum output power level
int fnLMS_GetMaxPwr(DEVID deviceID)
{

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    return lms[deviceID].MaxPower;
}

// Get the minimum output power level
int fnLMS_GetMinPwr(DEVID deviceID)
{

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    return lms[deviceID].MinPower;
}

// Get the relative output power level setting
int fnLMS_GetPowerLevel(DEVID deviceID)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    // Update our stored power level value
    if (!TestMode) {
        GetParameter(deviceID, VNX_PWR);
    }

    return lms[deviceID].Power;
}

// Get the absolute output power level setting
int fnLMS_GetAbsPowerLevel(DEVID deviceID)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID;
    }

    if (!isDeviceSockOpen(deviceID)) {
        return DEVICE_NOT_READY;
    }

    return (lms[deviceID].MaxPower - lms[deviceID].Power);
}


// Get the maximum frequency 0 to 40 GHz
unsigned int fnLMS_GetMaxFreq(DEVID deviceID)
{

    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID_EX;
    }

    return lms[deviceID].MaxFrequency;
}

// Get the minimum frequency, 0 to 40 GHz
unsigned int fnLMS_GetMinFreq(DEVID deviceID)
{
    if (deviceID >= MAXDEVICES) {
        return INVALID_DEVID_EX;
    }

    return lms[deviceID].MinFrequency;
}

