// --------------------------------- blxsocket.h -------------------------------------------
//
//  Include file for Linux LabBrick BLX socket definitions
//
//  (c) 2020-2025 by Vaunix Technology Corporation, all rights reserved
//
//  NJB  Version 1.0 BLX Ethernet Socket Device Driver
//
//-----------------------------------------------------------------------------
#ifndef __BLXSOCKET_H__
#define __BLXSOCKET_H__

/// ---------- Include headers ----------------
#include "blxdrvr.h"

//*****************************************************************************
//
// If building with a C++ compiler, make all of the definitions in this header
// have a C binding.
//
//*****************************************************************************
#ifdef __cplusplus
extern "C"
{
#endif

// --------- hardware device types used internally in the DLL ---------- 
#define HW_BLX403 21	// the hardware id for the BLX-403
#define HW_BLX403_20 23 // the hardware id for the BLX-403-20
#define HW_BLX223 24 // the hardware id for the BLX-403-20

#define BYTE unsigned char

// Format of the status report, for the convenience of host side coders
typedef struct
{
	BYTE pkt_status;
	BYTE count;
	unsigned int frequency;
	BYTE dev_status;
	unsigned char power;
} VNX_STATUS_REPORT;

#define STATUS_PLL_LOCK 0x80		// MASK: Pll lock status bit, 1 = locked
#define STATUS_NEW_PARAM 0x40		// MASK: A parameter was set since the last "Save Settings" command
#define STATUS_CMD_OK 0x20			// MASK: The command was completed by the LMS device without problems
									//		 !!! Note !!! renamed but value unchanged to avoid a name collision
									//		 with the status return code for the DLL itself.
#define STATUS_RF_ON 0x10			// MASK: The RF HW is on (moved from 0x08


// Bit masks and equates for the Sweep command byte (stored in Sweep_mode, and reported also in Status)
#define SWP_BIDIR			0x08	// MASK: bit = 0 for ramp style sweep, 1 for triangle style sweep								
#define SWP_DIRECTION		0x04	// MASK: bit = 0 for sweep up, 1 for sweep down 
#define SWP_CONTINUOUS		0x02	// MASK: bit = 1 for continuous sweeping
#define SWP_ONCE			0x01	// MASK: bit = 1 for single sweep

// equates for the Sequence variation of the sweep command byte
#define SEQ_STOP			0		// STOP for both sweeps and sequences
#define SEQ_ONCE			0x81	// SWP_ONCE | SWP_SEQUENCE_ACTIVE
#define SEQ_REPEAT			0x82	// SWP_CONTINUOUS | SWP_SEQUENCE_ACTIVE
#define SEQ_PAUSE			0xA0	// SWP_PAUSE | SWP_SEQUENCE_ACTIVE (or in the once or repeat bit)
#define SEQ_RESUME			0x03    // SWP_RESUME (we really don't need the sequence active bit!)

// Bit masks and equates for the DSS Pulse mode command byte (stored in Pulse_mode)
#define PM_ACTIVE			0x01	// MASK: bit = 0 for normal mode, 1 for pulse mode
#define PM_EXTERNAL			0x02	// MASK: bit = 0 for normal mode, 1 for external modulation

// HID report equates
#define HR_BLOCKSIZE 6				// size of the block of bytes buffer in our HID report
#define HID_REPORT_LENGTH 8			// use an 8 byte report..

// Misc. constants used for the Vaunix devices
#define MAX_ATTEN 64			// we've got 64 db max attenuation -- we represented it as 4 * the value
								// or the attentuation in .25db steps
#define BLX_MIN_DWELLTIME 10	// 10 ms. minimum dwell time for the BLX-403

// Misc commands to send to the device
#define VNX_SET				0x80
#define VNX_GET				0x00	// the set and get bits are or'd into the msb of the command byte

#define VNX_FREQUENCY		0x44	// output frequency in 100Khz units
#define VNX_FDWELL			0x05	// time to dwell at each frequency in the sweep, in 1ms units, minimum of 10ms.
#define VNX_FSTART			0x46	// frequency lower limit of sweep in 100Khz units 
#define VNX_FSTOP			0x47	// frequency upper limit of sweep in 100Khz units
#define VNX_FSTEP			0x08	// frequency step size, in 100Khz units except for BLX-403 which uses 10Hz units
#define VNX_SWEEP			0x09	// command to start/stop sweep, data = 01 for single sweep, 00 to stop
									// sweeping, and 02 for continuous sweeping.
									// For the DSS synth, there are more sweep option bits

#define VNX_OFFTIME			0x49	// length of pulse mode RF off time in microseconds
#define VNX_ONTIME			0x4A	// length of pulse mode RF on time in microseconds
#define VNX_PULSE_MODE		0x4B	// start/stop pulse mode

#define VNX_RFMUTE			0x0A	// enable or disable RF output, byte = 01 to enable, 00 to disable
#define VNX_ATTEN			0x0B	// attenuator setting, byte = number of dbm attenuation in .25 db steps
									// 00 is no attenuation, 02 is .5 db, 04 is 1 db

#define VNX_SAVEPAR			0x0C	// command to save user parameters to flash, data bytes must be
									// set to 0x42, 0x55, 0x31 as a key to enable the flash update
									// all of the above settings are saved (Frequency, sweep params, etc.

#define VNX_PWR				0x0D	// power output setting, relative to calibrated value - adds to calibrated
									// attenuator setting. It is a byte, with the attenuation expressed in .25db
									// steps. Thus, 2 = .5 db attenuation, 4 = 1.0db attenuation, etc. 									

#define VNX_DEFAULTS		0x0F	// restore all settings to factory default
									// FSTART = Minimum Frequency, FSTOP = Maximum Frequency
									// SWEEPTIME = ??, FSTEP = ??
									// FREQUENCY = ??

#define VNX_DSS_STATUS			0x4E	// Not really a command, but the status byte value for periodic status reports.	

#define VNX_INTOSC			0x23	// enable/disable internal reference oscillator - Byte == 01 for on, 00 for off
#define VNX_AIDLE			0x36	// idle time between sweeps in milliseconds

#define VNX_EXTSWP			0x4C    // Use Internal Sweep Trigger = 00, 01 = use external signal to trigger sweep

#define VNX_SETPROFILE		0x3A	// set/get profile values, first byte is unused
									// second data byte is the index (0 based)
									// the third is the attenuation value for that profile entry

#define VNX_PROFILECOUNT	0x3B	// number of elements in the profile, 1 to PROFILE_MAX = 50

#define VNX_PROFILEDWELL	0x3C	// dwell time for each profile element

#define VNX_PROFILEIDLE		0x3D	// idle time at the end of each repeating profile
#define VNX_PROFILESTART    0x3E    // the starting index for the profile (or sequence on a BLX-403)

// ---------------------- Factory commands ---------------------------

#define VNX_SAVECAL			0x1D	// command to write the calibration table to flash memory
									// data bytes must be set to 0xFA, 0x89, 0x73 as a key to enable
									// the flash update
#define VNX_SAVESERNUM		0x1E	// command to write the device's serial number to flash
									// data bytes must be set to 0xFA, 0x89, 0x73 as a key to enable
									// the flash update
#define VNX_SETSERNUM		0x1F	// set/get the serial number, value is a DWORD
#define VNX_SETCAL			0x1A	// set/get calibration values, first byte is the VCO bank (0 or 1)
									// second data byte is the index (0 based)
									// the third is the attenuation value
#define VNX_CALPWR			0x1B	// set/get the calibrated power output, value is in the first data byte
									// this is saved by the SAVECAL command
#define VNX_MAX_PWR			0x1B	// just an alias to VNX_CALPWR

#define VNX_MINFREQUENCY	0x20	// get (no set allowed) the minimum frequency
#define VNX_MAXFREQUENCY	0x21	// get (no set allowed) the maximum frequency

#define VNX_MODELNAME		0x22	// get (no set allowed) the device's model name string -- last 6 chars only

// ---------------------- BLX system Specific Commands ---------------------
#define VNX_IPMODE         0x61     // IP interface mode Static/DHCP
#define VNX_IPADDR         0x62     // IP Address
#define VNX_IPMASK         0x63     // Subnet Mask
#define VNX_IPGW           0x64     // Gateway

// Device IP length (IPV4)
#define DEVICE_IPV4_LENGTH 16
#define RXBUFF_DATA_LENGTH 24
#define TXBUFF_CMD_LENGTH   8

// Response Data Message Datastructure
typedef struct
{
  char command;
  char length;
  char data[HR_BLOCKSIZE];
} RESPONSE_DATA_T;

// BLX Device Response Data Structure
typedef struct
{
	int	DevStatus;

	int ipmode;
	char ipaddress[MAX_NETBUFF];
	char netmask[MAX_NETBUFF];
	char gateway[MAX_NETBUFF];

	unsigned int Frequency;
	unsigned int MinFrequency;
	unsigned int MaxFrequency;
	unsigned int StartFrequency;
	unsigned int EndFrequency;
	unsigned int FrequencyStep;

	int DwellTime;
	int SweepIdle;               // The BLX-403 stepped sweeps have an idle period
	int Power;
	int MaxPower;
	int MinPower;
	int MaxAttenuation;
	int PulseOnTime;
	int PulseOffTime;
	float PulseOnTimeSeconds;
	float PulseOffTimeSeconds;
	bool HasOldBLX_FW;			// set true when we think we have an old BLX device

	unsigned int SequenceLength;
	unsigned int SequenceStart;
	unsigned int SequenceDwell;
	unsigned int SequenceIdle;
	unsigned int SequenceMaxLength; // the BLX-403 has a 50 element maximum sequence length
	// that is likely to remain a constant for a while
	bool ProfileActive;
	int SequenceMode;

	int Modebits;
	int DevType;
	int SerialNumber;
	char ModelName[MAX_MODELNAME];
	int MyDevID;

	BLX_SEQUENCE Sequence[SEQ_MAX_ENTRIES];				// storage for the sequence

	// Internal variables used to identify and manage the hardware
	char deviceip[DEVICE_IPV4_LENGTH];
	char sndbuff[TXBUFF_CMD_LENGTH];
	char rcvbuff[RXBUFF_DATA_LENGTH];
	int  devicesockfd;
	int  deviceready; // Set the device ready flag 
	int  responseready; // Response ready flag

} LMSPARAMS;

//*****************************************************************************
//
// Mark the end of the C bindings section for C++ compilers.
//
//*****************************************************************************
#ifdef __cplusplus
}
#endif

#endif // __BLXSOCKET_H__
