// --------------------------------- blxdrvr.h -------------------------------------------
//
//  Include file for Linux LabBrick BLX socket definitions
//
//  (c) 2020-2025 by Vaunix Technology Corporation, all rights reserved
//
//  NJB  Version 1.0 BLX Ethernet Driver Interface Definitions
//-----------------------------------------------------------------------------
#ifndef __BLXDRVR_H__
#define __BLXDRVR_H__

// ---------- Include headers ----------------

//*****************************************************************************
//
// If building with a C++ compiler, make all of the definitions in this header
// have a C binding.
//
//*****************************************************************************
#ifdef __cplusplus
extern "C"
{
#endif


/// ---------- Global Equates ----------------
#define MAXDEVICES 64
#define MAX_MODELNAME   32
#define SEQ_MAX_ENTRIES 50
#define MAX_SWVERSION   7
#define MAX_NETBUFF     16

#define F10HZ_21GHZ 2100000000		// 21 GHz limit for 32 bit ints using 10Hz units
#define BLX_HAS_SWEEP 30700			// Firmware serial number for full featured firmware
#define BLX_SEQ_EMPTY 0				// we mark empty sequence elements with a zero value in their frequency field

// ----------- Data Types -----------------
#define DEVID unsigned int

typedef struct
{
	unsigned int Frequency10Hz;     // output frequency in 10Hz units
	unsigned char PowerLevel;		// power level in 0.25db units of attenuationn from max power
	bool ControlPower;				// true if the power level is used, false to ignore the power setting
}BLX_SEQUENCE;

// ----------- Mode Bit Masks ------------
#define MODE_INTREF		0x00000020			// bit is 1 for internal osc., 0 for external reference
#define MODE_RFON		0x00000040			// bit is 1 for RF on, 0 if RF is off (moved from 0x10 by RD 5/23/21)

#define MODE_SWEEP		0x0000000F			// bottom 4 bits are used to keep the sweep control bits
                                            // the bit definitions are in LMSHid.h

#define MODE_PWMON		0x00000100			// we keep a copy of the PWM control bits here, 1 for int PWM on
#define MODE_EXTPWM		0x00000200			// 1 for ext. PWM input enabled
#define PWM_MASK		0x00000300
#define MODE_EXTSWTRG	0x00000400			// 1 for ext. sweep trigger enabled

// ----------- Sequence Control -----------
#define STOP_SEQUENCE 0						// stop a sequence
#define START_SEQUENCE 1					// start a sequence to run once
#define REPEAT_SEQUENCE 2					// start a repeating sequence
#define PAUSE_SEQUENCE 3					// pause a running sequence
#define RESUME_SEQUENCE 4					// resume a running sequence

// Status returns for commands
#define LVSTATUS int

#define STATUS_OK 0
#define INVALID_DEVID			0x80000000	// MSB is set if the device ID is invalid
#define BAD_PARAMETER			0x80010000	// out of range input -- frequency outside min/max etc.
#define BAD_HID_IO				0x80020000	// a failure in the I/O subsystem
#define DEVICE_NOT_READY		0x80030000	// device isn't open, no handle, etc.
#define FEATURE_NOT_SUPPORTED	0x80040000	// the selected Lab Brick does not support this function
#define DEV_TABLE_FULL			0x80050000  // the device table is full
#define BAD_IP_ADDRESS			0x80060000	// the IP address is invalid (e.g. wrong format, etc)

// Status returns for EX commands that use unsigned 32 bit integers
// These values are above the representation for 42 GHz in 10 Hz units
#define STATUS_OK_EX 0
#define INVALID_DEVID_EX	0xFFFF0000		// if the device ID is invalid, 4,294,901,760 decimal, 
#define BAD_PARAMETER_EX	0xFFFF0001		// out of range input -- frequency outside min/max etc.
#define BAD_HID_IO_EX		0xFFFF0002
#define DEVICE_NOT_READY_EX	0xFFFF0003		// device isn't open, no handle, etc.

#define F_INVALID_DEVID		-1.0			// for functions that return a float
#define F_DEVICE_NOT_READY	-3.0
#define F_BAD_PARAMETER		-4.0

// Status returns for DevStatus

#define DEV_CONNECTED		0x00000001		// LSB is set if a device is connected
#define DEV_OPENED			0x00000002		// set if the device is opened
#define SWP_ACTIVE			0x00000004		// set if the device is sweeping
#define SWP_UP				0x00000008		// set if the device is sweeping up in frequency
#define SWP_REPEAT			0x00000010		// set if the device is in continuous sweep mode
#define SWP_BIDIRECTIONAL	0x00000020		// set if the device is in bi-directional sweep mode

#define PLL_LOCKED			0x00000040		// set if the PLL lock status is TRUE (both PLL's are locked)
#define	FAST_PULSE_OPTION	0x00000080		// set if the fast pulse mode option is installed

// Internal values in DevStatus
#define DEV_LOCKED	  0x00002000		// set if we don't want read thread updates of the device parameters
#define DEV_RDTHREAD  0x00004000		// set when the read thread is running
#define DEV_BUFEMPTY  0x00008000		// set when the device buffers are empty

// Flags to encode pulse mode time ranges
#define PM48Mhz				0x10000000		// used to select the 48Mhz pulse mod clock
#define PM1Mhz				0x00000000		// used to select the 1Mhz pulse mod clock or sw pulsing
#define PM120MHz			0x40000000		// used to select the 120Mhz pulse mode clock



// LMS Test mode
void fnLMS_SetTestMode(bool testmode);
// Add LMS Device to list of available devices
int fnLMS_AddLMSDevice(char* deviceIP);
// Remove LMS Device from list of available devices
int fnLMS_RemoveLMSDevice(char* deviceIP);
// Get the number of available devices
int fnLMS_GetNumDevices(void);
// Obtain an array of device IDs
int fnLMS_GetDevInfo(DEVID* ActiveDevices);
// InitDevice
int fnLMS_InitDevice(DEVID deviceID);
// Close the Device Socket
int fnLMS_CloseDevice(DEVID deviceID);
// Device Ready
int fnLMS_CheckDeviceReady(DEVID deviceID);
// Get Model Name (ASCII clients)
int fnLMS_GetModelNameA(DEVID deviceID, char* Model_Name);
// Get Model Name (Unicode clients)
int fnLMS_GetModelNameW(DEVID deviceID, wchar_t* Model_Name);
// Get Serial Number of the device
int fnLMS_GetSerialNumber(DEVID deviceID);
// Get SW Version of the device
int fnLMS_GetLibVersion(void);
// Get the status code of the device
int fnLMS_GetDeviceStatus(DEVID deviceID);
// Get IP Mode of the device  0 - Static, 1 - DHCP
LVSTATUS fnLMS_GetIPMode(DEVID deviceID, int* respdata);
// Get IP Address of the device
LVSTATUS fnLMS_GetIPAddress(DEVID deviceID, char* respdata);
// Get Netmask of the device
LVSTATUS fnLMS_GetNetmask(DEVID deviceID, char* respdata);
// Get Gateway of the device
LVSTATUS fnLMS_GetGateway(DEVID deviceID, char* respdata);


// Set Frequency in 10 Hz units
LVSTATUS fnLMS_SetFrequency(DEVID deviceID, unsigned int frequency);

// Set sweep start frequency in 10 Hz units
LVSTATUS fnLMS_SetStartFrequency(DEVID deviceID, unsigned int startfrequency);
// Set sweep end frequency in 10 Hz units
LVSTATUS fnLMS_SetEndFrequency(DEVID deviceID, unsigned int endfrequency);
// Set frequency step size in 10 Hz units
LVSTATUS fnLMS_SetFrequencyStep(DEVID deviceID, unsigned int frequency);
// Set sweep dwell time in ms
LVSTATUS fnLMS_SetDwellTime(DEVID deviceID, int dwelltime);
// Set sweep idle time in ms
LVSTATUS fnLMS_SetIdleTime(DEVID deviceID, int idletime);

// Set absolute power level in 0.25 dB units
LVSTATUS fnLMS_SetPowerLevel(DEVID deviceID, int powerlevel);
// Set the RF output on or off
LVSTATUS fnLMS_SetRFOn(DEVID deviceID, bool on);

// Set the internal pulse modulation pulse on time in seconds
LVSTATUS fnLMS_SetPulseOnTime(DEVID deviceID, float pulseontime);
// Set the internal pulse modulation pulse off time in seconds
LVSTATUS fnLMS_SetPulseOffTime(DEVID deviceID, float pulseofftime);
// Turn on or off the internal pulsed mode output control.
LVSTATUS fnLMS_EnableInternalPulseMod(DEVID deviceID, bool on);
// Set the state of the pulsed mode output control - internal or external
LVSTATUS fnLMS_SetUseExternalPulseMod(DEVID deviceID, bool external);
// Set pulsed output parameters, pulse on time in seconds and pulse rep time in seconds, and start pulse
LVSTATUS fnLMS_SetFastPulsedOutput(DEVID deviceID, float pulseontime, float pulsereptime, bool on);

// Set the state of the frequency reference - internal or external
LVSTATUS fnLMS_SetUseInternalRef(DEVID deviceID, bool internal);
// Set the sweep direction -- "up" is TRUE to sweep upwards
LVSTATUS fnLMS_SetSweepDirection(DEVID deviceID, bool up);
// Set the sweep mode -- mode = TRUE for repeated sweep, FALSE for one time sweep
LVSTATUS fnLMS_SetSweepMode(DEVID deviceID, bool mode);
// Set the sweep type -- swptype =	TRUE for bi-directional, FALSE for uni-directional sweep
LVSTATUS fnLMS_SetSweepType(DEVID deviceID, bool swptype);
// Start the sweep
LVSTATUS fnLMS_StartSweep(DEVID deviceID, bool go);
// Set the state of the sweep trigger control - internal or external
LVSTATUS fnLMS_SetUseExternalSweepTrigger(DEVID deviceID, bool external);

// set an element in the sequence, index runs from 0 to 49, frequency is in 10Hz units, powerlevel is in 0.25 db units (absolute power level)
LVSTATUS fnLMS_SetSequenceElement(DEVID deviceID, int index, unsigned int frequency, int powerlevel, bool pwr_control);
// Set the starting element of the sequence (zero based)
LVSTATUS fnLMS_SetSequenceStart(DEVID deviceID, int start);
// Set the length of the sequence
LVSTATUS fnLMS_SetSequenceCount(DEVID deviceID, int count);
// Set the dwell time for each element of the sequence
LVSTATUS fnLMS_SetSequenceDwellTime(DEVID deviceID, int dwelltime);
// Set the idle time at the end of the sequence
LVSTATUS fnLMS_SetSequenceIdleTime(DEVID deviceID, int idletime);
// Control the operation of sequences - stop=0, start=1, repeat=2, pause=3, resume=4
LVSTATUS fnLMS_StartSequence(DEVID deviceID, int control);

// Save the user settings to flash for autonomous operation
LVSTATUS fnLMS_SaveSettings(DEVID deviceID);

// Get the frequency in 10Hz units
unsigned int fnLMS_GetFrequency(DEVID deviceID);

// Get the sweep start frequency in 10Hz units
unsigned int fnLMS_GetStartFrequency(DEVID deviceID);
// Get the sweep end frequency in 10Hz units
unsigned int fnLMS_GetEndFrequency(DEVID deviceID);
// Get the frequency step size in 10Hz units for the stepped sweep.
unsigned int fnLMS_GetFrequencyStep(DEVID deviceID);
// Get the dwell time per step in 1ms units for stepped sweeps
int fnLMS_GetDwellTime(DEVID deviceID);
// Get the idle time in 1ms units for repeated stepped sweeps
int fnLMS_GetIdleTime(DEVID deviceID);

// Get the state of the RF output - on or off
int fnLMS_GetRF_On(DEVID deviceID);
// Get the state of the frequency reference - internal or external
int fnLMS_GetUseInternalRef(DEVID deviceID);
// Get the relative output power level setting in 0.25 dB units
int fnLMS_GetPowerLevel(DEVID deviceID);
// Get the absolute output power level setting in 0.25 dB units
int fnLMS_GetAbsPowerLevel(DEVID deviceID);
// Get the maximum output power level in 0.25 dB units
int fnLMS_GetMaxPwr(DEVID deviceID);
// Get the minimum output power level in 0.25 dB units
int fnLMS_GetMinPwr(DEVID deviceID);

// Get the maximum frequency in 10Hz units
unsigned int fnLMS_GetMaxFreq(DEVID deviceID);
// Get the minimum frequency in 10Hz units
unsigned int fnLMS_GetMinFreq(DEVID deviceID);

// Get the pulse on time in seconds
float fnLMS_GetPulseOnTime(DEVID deviceID);
// Get the pulse off time in seconds
float fnLMS_GetPulseOffTime(DEVID deviceID);
// Get if pulse mode output is on or off
int fnLMS_GetPulseMode(DEVID deviceID);

// Get if the fast pulse mode option is installed
int fnLMS_GetHasFastPulseMode(DEVID deviceID);
// Get if we are using internal pulse modulation source
int fnLMS_GetUseInternalPulseMod(DEVID deviceID);
// Get if we are using internal sweep trigger source
int fnLMS_GetUseInternalSweepTrigger(DEVID deviceID);

// Get the frequency in 10Hz units of a specified sequence element
unsigned int fnLMS_GetSeqElementFrequency(DEVID deviceID, int index);
// Get the absolute power in 0.25 db units of a specified sequence element
int fnLMS_GetSeqElementPower(DEVID deviceID, int index);
// Get if power is controlled or not by a specified sequence element
int fnLMS_GetSeqElementPwrControl(DEVID deviceID, int index);
// Get the starting element index of the sequence
int fnLMS_GetSequenceStart(DEVID deviceID);
// Get the length of the sequence
int fnLMS_GetSequenceCount(DEVID deviceID);
// Get the dwell time per sequence element in 1ms units
int fnLMS_GetSequenceDwellTime(DEVID deviceID);
// Get the idle time in 1ms units for repeated sequences
int fnLMS_GetSequenceIdleTime(DEVID deviceID);

//*****************************************************************************
//
// Mark the end of the C bindings section for C++ compilers.
//
//*****************************************************************************
#ifdef __cplusplus
}
#endif

#endif // __LDADRVR_H__
