// BLXEthTest.c : This file contains the 'main' function. Program execution begins and ends there.
//
// This version is an example CLI for Linux
#include <stdbool.h>
#include <stdio.h>
#include <stdint.h>         
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include <wctype.h>
#include <wchar.h>
#include <unistd.h>
#include <time.h>
#include "blxdrvr.h"


// to enable the BLX sequence demo code 
#define BLX_TEST 1
// #define BLX_TEST 0

#define TRUE 1
#define FALSE 0

// IP Address Validator
#define DELIM "."

// ------------------------------- Equates -----------------------------------------------
#define VT_DWP_BIDIRECTIONAL	0x08	// MASK: bit = 1 for bi-directional sweeping
#define VT_SWP_DIRECTION		0x04	// MASK: bit = 0 for sweep up, 1 for sweep down 
#define VT_SWP_CONTINUOUS		0x02	// MASK: bit = 1 for continuous sweeping
#define VT_SWP_ONCE				0x01	// MASK: bit = 1 for single sweep

// ------------------------------- Allocations -------------------------------------------
static DEVID MyDevices[MAXDEVICES];			// I have statically allocated this array for convenience
// It holds a list of device IDs for the connected devices
// They are stored starting at MyDevices[0]

static char MyDeviceNameA[MAX_MODELNAME];		// NB -- this is a single byte char array for testing the ASCII name function

static char errmsg[32];					// For the status->string converter
static char cModelName[32];					// buffer for the model name

static char* sDevName = "BLX-403";			// device name string
static bool gbWantOneDevice = FALSE;

static char* ndeviceip;
static long DevNum = 0;						// which device we should work with, default 0

static unsigned long Frequency = 500000000;	// default test frequency 5GHz
static unsigned long FStart = 500000000;	// default test start frequency
static unsigned long FStop = 600000000;		// default test stop is 6Ghz
static unsigned long FStep = 10000000;		// default test step is 100Mhz

static int DwellTime = 100;					// default dwell time 100 ms
static int IdleTime = 1000;					// default idle time 1 second

static long Atten = 0;						// default attenuation is 0db, encoding is in .25 db steps

static char* sDevPower = "";
static long Power = 4;						// sets output power level, relative to device's max calibrated power
											// it is in units of .25 db of attenuation - 00 is the largest signal

static long SerialNumber = 0;				// used to hold the serial number for the get serial number command

static int RefOsc = 0;						// really used as a bool -- if non zero, then we use the internal osc.

static int RFOnOff = 1;						// really used as a bool -- if non zero, turn on the RF output

static int Sweep_mode = 0;					// a variable to hold the user's desired sweep mode

static float PulsePeriod = 1.0;				// start with 1 second pulse period. Minimum = 1.0e-7
static float PulsePercent = 100;			// percentage on time, 100 disables pulse mode

static int IntTrigger = 1;					// really used as a bool -- if non zero select internal trigger source

static int SequenceCount = 0;
static int SequenceStart = 0;

static char gblxdevicesip[16];       // devices ip address

bool gbDeviceOpen = FALSE;
bool gbWantSetFrequency = FALSE;
bool gbWantSetFStart = FALSE;
bool gbWantSetFStop = FALSE;
bool gbWantSetFStep = FALSE;
bool gbWantSetDwell = FALSE;
bool gbWantSetIdle = FALSE;
bool gbWantStartSweep = FALSE;
bool gbWantSetAtten = FALSE;
bool gbWantSetPower = FALSE;
bool gbWantSaveSettings = FALSE;
bool gbWantSetSerialNum = FALSE;
bool gbQuietMode = FALSE;
bool gbGotReply = FALSE;
bool gbWantMaxPower = FALSE;
bool gbBatchMode = FALSE;
bool gbWantSetRefOsc = FALSE;
bool gbWantSetRFOnOff = FALSE;
bool gbWantSetPulsedOutput = FALSE;
bool gbWantSetExtTrigger = FALSE;


// ------------------------------- Support Routines --------------------------------------

void PrintHelp()
{

	printf("\n");
	printf(" --- Overall modes and device selection. Defaults to first device if device not selected ---\n");
	printf("\n");
	printf("  -b        Batch mode, exit after sending commands\n");
	printf("  -q		Quiet mode, skip most outputs.\n");
	printf("  -d i 		Select the device to work with, i is the device ip address (192.168.100.11)\n");
	printf("\n");

	printf(" --- Commands to set parameters and start sweep --- \n");

	printf("  -f nn     Set frequency, nn is frequency in 1 Hz (1.0e9 for 1GHz) units\n");
	printf("  -s nn     Set sweep start frequency, nn is start frequency in native 10 Hz units\n");
	printf("  -e nn     Set sweep end frequency, nn is end frequency in native 10 Hz units\n");
	printf("  -i nn     Set BLX sweep increment, nn is step size in native 10 Hz units\n");
	printf("  -t nn     Set time to dwell on each frequency value, nn is time in ms.\n");
	printf("  -w nn     Set idle time between sweeps, nn is time in ms.\n");
	printf("  -p nn     Set output power, nn is output power in dBm\n");
	printf("  -g n      Start a sweep, 1 = once upwards, 2 = continuous upwards\n");
	printf("             5 = once down, 6 = continuous down, 0 = end sweep\n");
	printf("             9 = single bi-directional sweep, 10 = continuous bi-directional sweep\n");
	printf("  -o n      Select the reference oscillator to use -- 1 = internal, 0 = external\n");
	printf("  -r n      Turn the RF output on or off -- 1 = on, 0 = off\n");
	printf("\n");
	printf("  -m nn     Set the percentage on time for pulsed output - 100 disables pulsed output mode\n");
	printf("  -n nn     Pulse period repetition period in seconds. -n 1.0 -m 50 results in a 1 Hz 50 percent duty cycle\n");
	printf("\n");
	printf("  -y        Write user settings to flash\n");
	printf("            Hit CTRL+C to exit\n");
	printf("\n");
	printf("\n");

}


//***************************************************************************
//
// Function call for delay loop
//
//*****************************************************************************
/* usleep is deprecated, so this is a function using nanosleep() */
void sleep_ms(long naptime) {
  // naptime comes in as the number of ms we want. 20 = 20ms
  struct timespec timeval;
   timeval.tv_sec = 0;
   timeval.tv_nsec = naptime * 1000000L;

   nanosleep(&timeval , NULL);
}

//***************************************************************************
//
// Validate netconfig digits and return 1 if string contain only digits, else return 0
//
//*****************************************************************************
static int valid_digit(char* ip_str) {
	while (*ip_str) {
		if (*ip_str >= '0' && *ip_str <= '9')
			++ip_str;
		else
			return 0;
	}
	return 1;
}

//***************************************************************************
//
// Validate IP Configuration -return 1 if IP string is valid, else return 0
//
//*****************************************************************************
static int is_valid_ip(char* ip_str) {
	int num, dots = 0;
	char* ptr;
	char lstr[16];
	char* next_token;

	strcpy(lstr, ip_str);

	if (lstr == NULL)
		return 0;

	ptr = strtok(lstr, DELIM);

	if (ptr == NULL)
		return 0;

	while (ptr) {

		/* after parsing string, it must contain only digits */
		if (!valid_digit(ptr))
			return 0;

		num = atoi(ptr);

		/* check for valid IP */
		if (num >= 0 && num <= 255) {
			/* parse remaining string */
			ptr = strtok(NULL, DELIM);
			if (ptr != NULL)
				++dots;
		}
		else
			return 0;
	}

	/* valid IP string must contain 3 dots */
	if (dots != 3)
		return 0;
	return 1;
}

// --------------------- MakeLower ------------------------------


wchar_t MakeLowerW(wchar_t wc)
{
	return towlower(wc);
}

// --------------------------------------------------------------

#define MAX_MSG 32

// A function to display the status as a Unicode string
char* fnLMS_perror(LVSTATUS status) {
	strcpy(errmsg, "STATUS_OK");
	if (BAD_PARAMETER == status) strcpy(errmsg, "BAD_PARAMETER");
	if (BAD_HID_IO == status) strcpy(errmsg, "BAD_HID_IO");
	if (DEVICE_NOT_READY == status) strcpy(errmsg, "DEVICE_NOT_READY");
	if (INVALID_DEVID == status) strcpy(errmsg, "INVALID_DEVID");
	if (FEATURE_NOT_SUPPORTED == status) strcpy(errmsg, "FEATURE_NOT_SUPPORTED");
	if (DEV_TABLE_FULL == status) strcpy(errmsg, "DEV_TABLE_FULL");
	if (BAD_IP_ADDRESS == status) strcpy(errmsg, "BAD_IP_ADDRESS");

	// Status returns for DevStatus
	if (DEV_CONNECTED == status) strcpy(errmsg, "DEV_CONNECTED");
	if (DEV_OPENED == status) strcpy(errmsg, "DEV_OPENED");
	if (SWP_ACTIVE == status) strcpy(errmsg, "SWP_ACTIVE");
	if (SWP_UP == status) strcpy(errmsg, "SWP_UP");
	if (SWP_REPEAT == status) strcpy(errmsg, "SWP_REPEAT");
	if (SWP_BIDIRECTIONAL == status) strcpy(errmsg, "SWP_BIDIRECTIONAL");

	return errmsg;
}

// -- one way to check for errors --
bool CheckAPISet(LVSTATUS status)
{
	if (status & 0x80000000)
	{
		printf("*** Error: LMS API returned status = %x, %s ***\n", status, fnLMS_perror(status));
		return true;
	}
	else
	{
		return false;
	}
}

// A function to display the extended status as a Unicode string
char* fnLMS_perrorEx(unsigned int status) {
	strcpy(errmsg, "STATUS_OK");
	if (BAD_PARAMETER_EX == status) strcpy(errmsg, "BAD_PARAMETER");
	if (BAD_HID_IO_EX == status) strcpy(errmsg, "BAD_HID_IO");
	if (DEVICE_NOT_READY_EX == status) strcpy(errmsg, "DEVICE_NOT_READY");
	if (INVALID_DEVID_EX == status) strcpy(errmsg, "INVALID_DEVID");

	return errmsg;
}

// -- one way to check for errors from Ex functions --
bool CheckAPISetEx(unsigned int status)
{
	if ((status & 0xFFFF0000) == 0xFFFF0000)
	{
		printf("*** Error: LMS API returned status = %x, %s ***\n", status, fnLMS_perrorEx(status));
		return true;
	}
	else
	{
		return false;
	}
}

// A function to display the float status as a Unicode string
char* fnLMS_perrorF(unsigned int status) {
	strcpy(errmsg, "STATUS_OK");
	if (F_INVALID_DEVID == status) strcpy(errmsg, "INVALID_DEVID");
	if (F_DEVICE_NOT_READY == status) strcpy(errmsg, "DEVICE_NOT_READY");
	if (F_BAD_PARAMETER == status) strcpy(errmsg, "BAD_PARAMETER");

	return errmsg;
}

// -- one way to check for errors from Float functions --
bool CheckAPISetF(unsigned int status)
{
	if (status < 0)
	{
		printf("*** Error: LMS API returned status = %x, %s ***\n", status, fnLMS_perrorEx(status));
		return true;
	}
	else
	{
		return false;
	}
}



// -- convert the 10Hz integer units to MHz --
double Frequency_MHz(unsigned int frequency)
{
	return ((double)frequency) / 1.0e5;		// to get MHz we just divide the frequency in 10Hz units by 100,000
}

// ---------- ParseCommandLine ----------------------------------------------- 

// ParseCommandLine() will return FALSE to indicate that we received an invalid
// command or should abort for another reason.


bool ParseCommandLine(int argc, char *argv[])
{
	char devipstr[16];
	size_t icount;

	enum {
		wantDash, wantDevNumber, wantFrequency, wantFStart, wantFStop, wantFStep,
		wantDwell, wantIdle, wantAtten, wantPower, wantIndex, wantHiLo, wantSerialNum,
		wantSweep, wantMaxPower, wantSetRFOnOff, wantSetRefOsc, wantPulsePeriod,
		wantPulsePercent, wantExtTrigger
	} state = wantDash;

	for (int i = 1; i < argc; ++i) {
		char *arg = argv[i];
		size_t len = strlen(arg);
		// Convert argument to lowercase in-place
		for (size_t k = 0; k < len; ++k) {
			arg[k] = (char)tolower((unsigned char)arg[k]);
		}

		if (state == wantDash) {
			if (arg[0] != '-') {
				printf("\n *** Error in command line syntax *** \n");
				PrintHelp();
				return FALSE;
			}
			char *thisParam = arg + 1;

			if (strcmp(thisParam, "d") == 0) {
				state = wantDevNumber;
			} else if (strcmp(thisParam, "b") == 0) {
				gbBatchMode = TRUE;
			} else if (strcmp(thisParam, "f") == 0) {
				gbWantSetFrequency = TRUE;
				state = wantFrequency;
			} else if (strcmp(thisParam, "s") == 0) {
				gbWantSetFStart = TRUE;
				state = wantFStart;
			} else if (strcmp(thisParam, "e") == 0) {
				gbWantSetFStop = TRUE;
				state = wantFStop;
			} else if (strcmp(thisParam, "i") == 0) {
				gbWantSetFStep = TRUE;
				state = wantFStep;
			} else if (strcmp(thisParam, "t") == 0) {
				gbWantSetDwell = TRUE;
				state = wantDwell;
			} else if (strcmp(thisParam, "w") == 0) {
				gbWantSetIdle = TRUE;
				state = wantIdle;
			} else if (strcmp(thisParam, "a") == 0) {
				gbWantSetAtten = TRUE;
				state = wantAtten;
			} else if (strcmp(thisParam, "p") == 0) {
				gbWantSetPower = TRUE;
				state = wantPower;
			} else if (strcmp(thisParam, "g") == 0) {
				gbWantStartSweep = TRUE;
				state = wantSweep;
			} else if (strcmp(thisParam, "y") == 0) {
				gbWantSaveSettings = TRUE;
				state = wantDash;
			} else if (strcmp(thisParam, "q") == 0) {
				gbQuietMode = TRUE;
				state = wantDash;
			} else if (strcmp(thisParam, "r") == 0) {
				gbWantSetRFOnOff = TRUE;
				state = wantSetRFOnOff;
			} else if (strcmp(thisParam, "o") == 0) {
				gbWantSetRefOsc = TRUE;
				state = wantSetRefOsc;
			} else if (strcmp(thisParam, "m") == 0) {
				gbWantSetPulsedOutput = TRUE;
				state = wantPulsePercent;
			} else if (strcmp(thisParam, "n") == 0) {
				gbWantSetPulsedOutput = TRUE;
				state = wantPulsePeriod;
			} else if (strcmp(thisParam, "x") == 0) {
				gbWantSetExtTrigger = TRUE;
				state = wantExtTrigger;
			} else {
				PrintHelp();
				return FALSE;
			}
		} else {
			// Handle argument values
			switch (state) {
				case wantDevNumber:
					strncpy(devipstr, arg, sizeof(devipstr)-1);
					devipstr[sizeof(devipstr)-1] = '\0';
					if (is_valid_ip(devipstr)) {
						strcpy(gblxdevicesip, devipstr);
						gbDeviceOpen = TRUE;
					} else {
						printf("Invalid IP Address\n");
					}
					state = wantDash;
					break;
				case wantFrequency:
					Frequency = (unsigned int)(strtof(arg, NULL) / 10);
					state = wantDash;
					break;
				case wantFStart:
					FStart = strtoul(arg, NULL, 10);
					state = wantDash;
					break;
				case wantFStop:
					FStop = strtoul(arg, NULL, 10);
					state = wantDash;
					break;
				case wantFStep:
					FStep = strtoul(arg, NULL, 10);
					state = wantDash;
					break;
				case wantDwell:
					DwellTime = strtol(arg, NULL, 10);
					state = wantDash;
					break;
				case wantIdle:
					IdleTime = strtol(arg, NULL, 10);
					state = wantDash;
					break;
				case wantAtten:
					Atten = strtol(arg, NULL, 10);
					state = wantDash;
					break;
				case wantPower:
					Power = (int)(strtof(arg, NULL) * 4);
					state = wantDash;
					break;
				case wantSerialNum:
					SerialNumber = strtol(arg, NULL, 10);
					state = wantDash;
					break;
				case wantSweep:
					Sweep_mode = strtol(arg, NULL, 10);
					state = wantDash;
					break;
				case wantSetRFOnOff:
					RFOnOff = strtol(arg, NULL, 10);
					state = wantDash;
					break;
				case wantSetRefOsc:
					RefOsc = strtol(arg, NULL, 10);
					state = wantDash;
					break;
				case wantPulsePercent:
					PulsePercent = strtof(arg, NULL);
					state = wantDash;
					break;
				case wantPulsePeriod:
					PulsePeriod = strtof(arg, NULL);
					state = wantDash;
					break;
				case wantExtTrigger:
					IntTrigger = strtol(arg, NULL, 10);
					state = wantDash;
					break;
				default:
					break;
			}
		}
	}

	if (state != wantDash) {
		PrintHelp();
		return FALSE;
	}

	return TRUE;
}

// -------------------------------- Program Main -----------------------------------------

int main(int argc, char* argv[])
{
	int i, j, idx;
	int itemp;
	int itemp2;
	unsigned int utemp;
	bool bTemp;
	float ftemp;
	float fMaxPwr;
	float fMinPwr;
	double dtemp;
	float pmod_on, pmod_off, pmod_total;

	printf("Lab Brick BLX Test Program\n");

	itemp = fnLMS_GetLibVersion();
	printf("DLL Version = %x\n", itemp);


	if (!ParseCommandLine(argc, argv))
		return 0;


	// --- if TestMode = TRUE then the dll will fake the hardware ---
	fnLMS_SetTestMode(FALSE);

	// --- see if we have some hardware to use ---
	if (gbDeviceOpen) {
		fnLMS_AddLMSDevice(gblxdevicesip);
	}

	i = fnLMS_GetNumDevices();

	if (i == 0){
		printf("No device found\n");
	}
	else if (i < 0)
	{
		printf("GetNumDevices Returned %s\n", fnLMS_perror(i));		// an example of displaying API error returns
	}

	if (i == 1){
		if (!gbQuietMode) printf("Found %d Device\n", i);
	}
	else {
		if (!gbQuietMode) printf("Found %d Devices\n", i);
	}

	i = fnLMS_GetDevInfo(MyDevices);

	if (i < 0)
	{
		printf("GetDevInfo Returned %s\n", fnLMS_perror(i));		// an example of displaying API error returns
	}

	if (!gbQuietMode) printf("Got Device Info for %d Device[s]\n", i);


	if (i > 0)	// do we have a device? 
	{
		for (j = 0; j < i; j++) {
			printf("\n");

			// --- print out the first device's name ---
			if (!gbQuietMode) {
				itemp = fnLMS_GetModelNameA(MyDevices[j], MyDeviceNameA);
				printf("Device %d is an %s \n", MyDevices[j], MyDeviceNameA);
			}

			// --- print out the device's serial number ---
			if (!gbQuietMode) {
				itemp = fnLMS_GetSerialNumber(MyDevices[j]);
				printf("Device %d has serial number %d \n", MyDevices[j], itemp);
			}

			// --- We need to init the device (open it) before we can do anything else ---
			itemp = fnLMS_InitDevice(MyDevices[j]);

			if (itemp < 0)
			{
				printf("InitDevice Returned %s\n", fnLMS_perror(itemp));		// an example of displaying API error returns
			}

			if (itemp < 0)
			{
				printf("InitDevice Returned %s\n", fnLMS_perror(itemp));		// an example of displaying API error returns
			}

			// --- Lets see if we got the device's parameters ---
			if (!gbQuietMode) {

				utemp = fnLMS_GetFrequency(MyDevices[j]);
				printf("Frequency = %f Mhz ( %lu in 10Hz units)\n", Frequency_MHz(utemp), utemp);

				utemp = fnLMS_GetStartFrequency(MyDevices[j]);
				printf("Sweep Start Frequency = %f Mhz ( %lu in 10Hz units)\n", Frequency_MHz(utemp), utemp);

				utemp = fnLMS_GetEndFrequency(MyDevices[j]);
				printf("Sweep End Frequency = %f Mhz ( %lu in 10Hz units)\n", Frequency_MHz(utemp), utemp);

				utemp = fnLMS_GetMinFreq(MyDevices[j]);
				printf("Minimum Frequency =  %f Mhz ( %lu in 10Hz units)\n", Frequency_MHz(utemp), utemp);

				utemp = fnLMS_GetMaxFreq(MyDevices[j]);
				printf("Maximum Frequency =  %f Mhz ( %lu in 10Hz units)\n", Frequency_MHz(utemp), utemp);

				itemp = fnLMS_GetMinPwr(MyDevices[j]);
				fMinPwr = itemp * .25f;	// we represent power levels in .25db steps
				printf("Minimum Output Power Level = %.2f db\n", fMinPwr);

				itemp = fnLMS_GetMaxPwr(MyDevices[j]);
				fMaxPwr = itemp * .25f;	// we represent power levels in .25db steps
				printf("Maximum Output Power Level = %.2f db\n", fMaxPwr);


				// --- Show if the RF output is on ---
				itemp = fnLMS_GetRF_On(MyDevices[j]);

				if (itemp != 0) {
					printf("RF ON\n");
				}
				else {
					printf("RF OFF\n");
				}

				// --- Show Ref. Oscillator Source ---
				itemp = fnLMS_GetUseInternalRef(MyDevices[j]);

				if (itemp != 0) {
					printf("Using Internal Reference Frequency\n");
				}
				else {
					printf("Using External Reference Frequency\n");
				}

				// --- Show internal or external sweep trigger ---
				itemp = fnLMS_GetUseInternalSweepTrigger(MyDevices[j]);

				if (itemp > 0) {
					printf("Using Internal Sweep Trigger\n");
				}
				else if (itemp == 0) {
					printf("Using External Sweep Trigger\n");
				}
				else {
					printf("GetUseInternalSweep Returned %s\n", fnLMS_perror(itemp));		// an example of displaying API error returns
				}

				// --- Show internal vs. external Pulse Modulation Source ---
				itemp = fnLMS_GetUseInternalPulseMod(MyDevices[j]);

				if (itemp != 0) {
					printf("Using Internal Pulse Modulator\n");
				}
				else {
					printf("Using External Pulse Modulation Control\n");
				}

				// --- Show if Pulse Modulation is enabled ---
				itemp = fnLMS_GetDeviceStatus(MyDevices[j]);

				if (itemp & FAST_PULSE_OPTION) {
					printf("Pulse Modulation Option Present\n");
				}
				else {
					printf("Pulse Modulation Option Not Present\n");
				}

				// --- Show the present pulse modulation on and off times ---
				ftemp = fnLMS_GetPulseOnTime(MyDevices[j]);
				printf("Pulse On Time = %f sec\n", ftemp);

				ftemp = fnLMS_GetPulseOffTime(MyDevices[j]);
				printf("Pulse Off Time = %f sec\n", ftemp);

				// --- Show the present output power level using the relative power level function ---
				itemp = fnLMS_GetPowerLevel(MyDevices[j]);
				ftemp = itemp * .25f;	// we represent power levels in .25db steps
										// note that this function returns the power setting relative to Maximum Power
				printf("Output Power Level = %.2f db\n", (fMaxPwr - ftemp));

				// --- Show the state ofthe PLL Lock Indicator ---
				itemp = fnLMS_GetDeviceStatus(MyDevices[j]);

				if (itemp & PLL_LOCKED) {
					printf("PLL Locked\n");
				}
				else {
					printf("PLL is not in lock\n");
				}

				printf(" --------------------------------- \n");

				// -- BLX specific features --
				if (strstr(MyDeviceNameA, "BLX") != NULL)
				{
					// we have a BLX device, so we'll display the parameters for its features
					printf("\nBLX Specific Features\n");

					// frequency step for stepped sweeps
					utemp = fnLMS_GetFrequencyStep(MyDevices[j]);
					printf("Sweep Frequency Step = %f Mhz ( %lu in 10Hz units)\n", Frequency_MHz(utemp), utemp);

					// dwell time for sweep
					itemp = fnLMS_GetDwellTime(MyDevices[j]);
					printf("Sweep Dwell Time = %d ms\n", itemp);

					// idle time for sweep
					itemp = fnLMS_GetIdleTime(MyDevices[j]);
					printf("Sweep Idle Time = %d ms\n", itemp);

					// Parameters for sequences
					itemp = fnLMS_GetSequenceCount(MyDevices[j]);
					printf("Number of Sequence Entries = %d\n", itemp);
					SequenceCount = itemp;

					// defend our arrays
					if ((SequenceCount < 0) || (SequenceCount > 50))
					{
						SequenceCount = 1;
						printf("*** Invalid Sequence Count ***\n");
					}

					itemp = fnLMS_GetSequenceStart(MyDevices[j]);
					printf("Sequence Starting Index = %d\n", itemp);
					SequenceStart = itemp;

					itemp = fnLMS_GetSequenceDwellTime(MyDevices[j]);
					printf("Sequence Dwell Time = %f sec. ( %d in 1ms. units)\n", ((float)itemp/1000), itemp);

					itemp = fnLMS_GetSequenceIdleTime(MyDevices[j]);
					printf("Sequence Idle Time = %f sec. ( %d in 1ms. units)\n", ((float)itemp/1000), itemp);
					
					// show the entries in the sequence
					printf("Sequence elements:\n");
					for (idx = 0; idx < SequenceCount; idx++)
					{
						utemp = fnLMS_GetSeqElementFrequency(MyDevices[j], idx);
						itemp = fnLMS_GetSeqElementPwrControl(MyDevices[j], idx);
						if (CheckAPISetEx(utemp) || CheckAPISet(itemp))
						{
							break;	// quit if we get an error
						}

						itemp2 = fnLMS_GetSeqElementPower(MyDevices[j], idx);

						

						printf("Entry %d  Freq = %f Mhz Power = %f dBm Pwr Control = %d\n", idx, Frequency_MHz(utemp), ((float)itemp2 / 4), itemp);
					}

				}

			} // end of our quiet mode case

		} // end of the for loop over the devices


		// ------------- Now we'll set the requested device with new parameters -------------
		if (!gbQuietMode) printf("\n ----- Set Parameters ----- \n");

		if (gbWantSetFrequency)
		{

			if (!gbQuietMode) printf("Setting Frequency = %f Mhz ( %lu in 10 Hz units)\n", Frequency_MHz(Frequency), Frequency);
			itemp = fnLMS_SetFrequency(MyDevices[DevNum], Frequency);
			CheckAPISet(itemp);
		}

		// --- and then do whatever else the user requested ---
		if (gbWantSetFStart)
		{
			if (!gbQuietMode) printf("Setting Sweep Start Frequency = %f Mhz ( %lu in 10 Hz units)\n", Frequency_MHz(FStart), FStart);
			itemp = fnLMS_SetStartFrequency(MyDevices[DevNum], FStart);
			CheckAPISet(itemp);
		}

		if (gbWantSetFStop)
		{
			if (!gbQuietMode) printf("Setting Sweep Stop Frequency = %f Mhz ( %lu in 10 Hz units)\n", Frequency_MHz(FStop), FStop);
			itemp = fnLMS_SetEndFrequency(MyDevices[DevNum], FStop);
			CheckAPISet(itemp);
		}

		if (gbWantSetFStep)
		{
			if (!gbQuietMode) printf("Setting Sweep Step Size = %f Mhz ( %lu in 10 Hz units)\n", Frequency_MHz(FStep), FStep);
			itemp = fnLMS_SetFrequencyStep(MyDevices[DevNum], FStep);
			CheckAPISet(itemp);
		}

		if (gbWantSetDwell)
		{
			if (!gbQuietMode) printf("Setting Sweep Dwell Time = %d ms\n", DwellTime);
			itemp = fnLMS_SetDwellTime(MyDevices[DevNum], DwellTime);
			CheckAPISet(itemp);
		}

		if (gbWantSetIdle)
		{
			if (!gbQuietMode) printf("Setting Sweep Idle Time = %d ms\n", IdleTime);
			itemp = fnLMS_SetIdleTime(MyDevices[DevNum], IdleTime);
			CheckAPISet(itemp);
		}

		if (gbWantSetPower)
		{
			if (!gbQuietMode) printf("Setting Power Level = %d in .25 db units\n", Power);
			itemp = fnLMS_SetPowerLevel(MyDevices[DevNum], Power);	// note, this function uses absolute power level!
			CheckAPISet(itemp);
		}

		if (gbWantStartSweep)
		{

			// --- first we'll figure out what the user wants us to do ---
			if (Sweep_mode & VT_SWP_DIRECTION)
			{
				bTemp = FALSE;
			}
			else
			{
				bTemp = TRUE;
			}	// NB -- don't confuse these similarly named VT_ constants for the API constants!!

			itemp = fnLMS_SetSweepDirection(MyDevices[DevNum], bTemp);	// TRUE means sweep upwards for the API
			CheckAPISet(itemp);

			// --- and now we'll do the mode - one time sweep or repeated sweep ---

			if (Sweep_mode & VT_SWP_ONCE)
			{
				bTemp = FALSE;
			}
			else
			{
				bTemp = TRUE;
			}	// NB -- the flag is the command line arg is not the same as the API constant!!

			itemp = fnLMS_SetSweepMode(MyDevices[DevNum], bTemp);		// TRUE means repeated sweep for the API
			CheckAPISet(itemp);

			if (Sweep_mode & VT_DWP_BIDIRECTIONAL)
			{
				itemp = fnLMS_SetSweepType(MyDevices[DevNum], TRUE);
				CheckAPISet(itemp);
			}
			else
			{
				itemp = fnLMS_SetSweepType(MyDevices[DevNum], FALSE);
				CheckAPISet(itemp);
			}

			if (!Sweep_mode)
			{
				itemp = fnLMS_StartSweep(MyDevices[DevNum], FALSE);
				CheckAPISet(itemp);
			}
			else
			{
				if (!gbQuietMode) printf("Starting a Frequency Sweep\n");
				itemp = fnLMS_StartSweep(MyDevices[DevNum], TRUE);
				CheckAPISet(itemp);
			}
		}

		if (gbWantSetRFOnOff)
		{

			if (RFOnOff == 0)
			{
				bTemp = FALSE;
			}
			else
			{
				bTemp = TRUE;
			}

			itemp = fnLMS_SetRFOn(MyDevices[DevNum], bTemp);
			CheckAPISet(itemp);
		}

		if (gbWantSetRefOsc)
		{

			if (RefOsc == 0)
			{
				bTemp = FALSE;
			}
			else
			{
				bTemp = TRUE;
			}

			itemp = fnLMS_SetUseInternalRef(MyDevices[DevNum], bTemp);
			CheckAPISet(itemp);
		}

		// -------- An example of using pulsed mode output ------------
		if (gbWantSetPulsedOutput)
		{

			if (!gbQuietMode) printf("Setting Pulse Modulation Period = %f, On Time = %f %%\n", PulsePeriod, PulsePercent);

			if (PulsePercent > 99.75)
			{
				fnLMS_EnableInternalPulseMod(MyDevices[DevNum], FALSE);
			}
			else
			{
				pmod_total = PulsePeriod;
				pmod_on = ((.01f * PulsePercent) * PulsePeriod);
				pmod_off = pmod_total - pmod_on;

				itemp = fnLMS_SetFastPulsedOutput(MyDevices[DevNum], pmod_on, pmod_total, TRUE);
				CheckAPISet(itemp);

				//	printf("sent pmod_on = %f, pmod_off = %f\n", pmod_on, pmod_off);

			}
		}


#if BLX_TEST

		// -- Sequencs are a BLX specific feature --
		if (strcmp(MyDeviceNameA, "BLX-403") == 0 ||
			strcmp(MyDeviceNameA, "BLX-403-20") == 0 || 
			strcmp(MyDeviceNameA, "BLX-223") == 0)
		{
			// set some sequence elements
			fnLMS_SetSequenceElement(MyDevices[DevNum], 0, 50000000, 20, TRUE);		// 500 MHz at +5 dBm
			fnLMS_SetSequenceElement(MyDevices[DevNum], 1, 62000000, 10, TRUE);		// 620 MHz at +2.5 dBm
			fnLMS_SetSequenceElement(MyDevices[DevNum], 2, 85000000, -20, TRUE);	// 850 MHz at -5 dBm

			// set the length and starting element of the sequence
			fnLMS_SetSequenceStart(MyDevices[DevNum], 0);					// typically sequences start at the first element (index = 0)
																			// but they can be started at other elements

			fnLMS_SetSequenceCount(MyDevices[DevNum], 3);					// our example has 3 elements

			// set the dwell time at each element in the sequence
			fnLMS_SetSequenceDwellTime(MyDevices[DevNum], 100);				// 100 ms

			// set the idle time between repeats of a repeating sequence
			fnLMS_SetSequenceIdleTime(MyDevices[DevNum], 300);				// 300 ms

			// run the sequence
			fnLMS_StartSequence(MyDevices[DevNum], START_SEQUENCE);

			printf("\nStarted Test Sequence\n");
		}

#endif

		// --- do this last, since the user probably wants to save what he just set ---
		if (gbWantSaveSettings)
		{
			itemp = fnLMS_SaveSettings(MyDevices[DevNum]);
			CheckAPISet(itemp);
		}

		// -- The user wants us to exit right away --
		if (gbBatchMode)
		{
			for (j = 0; j < i; j++)
			{
				itemp = fnLMS_CloseDevice(MyDevices[j]);
				CheckAPISet(itemp);
			}
			return 0;		// we're done, exit to the command prompt
		}

		// -- Lets hang around some and report on the device's operation
		j = 0;
		while (j < 20)
		{
			// for the BLX-403 and similar devices that support frequencies above 20 GHz
			// the Ex functions are used

			utemp = fnLMS_GetFrequency(MyDevices[DevNum]);
			dtemp = ((double) utemp ) / 100000;
			printf("Frequency = %f MHz ( %lu in 10 Hz units)\n", dtemp, utemp);

			// -- using the GetAbsPowerLevel function to read our absolute output power level ---
			itemp = fnLMS_GetAbsPowerLevel(MyDevices[DevNum]);
			ftemp = itemp * .25f;	// we represent power levels in .25db steps
			printf("Output Power Level = %.2f dBm\n", ftemp);

			sleep_ms(500);		// wait for 1/2 second

			j++;
		}

		// -- we've done whatever the user wanted, time to close the devices
		if (!gbQuietMode) printf("Closing devices...\n");

		for (j = 0; j < i; j++)
		{
			itemp = fnLMS_CloseDevice(MyDevices[j]);
			CheckAPISet(itemp);
		}

	} // end of if ( i > 0 ) -- "we have a device"

	return 0;
}

